package com.intentsoftware.addapptr

import android.util.Log
import com.intentsoftware.addapptr.BannerCache.CacheDelegate
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.isLoggable

/**
 * Used to set up [BannerCache].
 */
class BannerCacheConfiguration {
    val placementName: String
    val size: Int

    /**
     * Sets optional cache delegate.
     */
    var delegate: CacheDelegate? = null
    private var shouldCacheAdditionalAdAtStart = false

    /**
     * The configuration that will be used when requesting new banners.
     */
    var requestConfiguration: BannerRequest? =
        BannerRequest(null) //prepare default if user does not need extra configuration

    var minDelayMillis = 1000
        private set

    /**
     * Creates the configuration object to be used when creating BannerCache
     *
     * @param placementName The name of the banner placement that will be created. **The placement will be created by the cache and should not be created manually.**
     * @param size          Defines how many preloaded banners should be available in the cache. Should be smaller than {@value MAX_CACHE_SIZE}.
     */
    @Suppress("NAME_SHADOWING")
    constructor(placementName: String, size: Int) {
        var size = size
        this.placementName = placementName
        if (size <= 0) {
            if (isLoggable(Log.ERROR)) {
                e(this, "Cache size cannot be lower than 1! Size = 1 will be used.")
            }
            size = 1
        } else if (size > MAX_CACHE_SIZE) {
            if (isLoggable(Log.ERROR)) {
                e(
                    this,
                    "Cache size cannot be greater than $MAX_CACHE_SIZE. Size = $MAX_CACHE_SIZE will be used."
                )
            }
            size = MAX_CACHE_SIZE
        }
        this.size = size
    }

    internal constructor(configuration: BannerCacheConfiguration) {
        placementName = configuration.placementName
        size = configuration.size
        delegate = configuration.delegate
        shouldCacheAdditionalAdAtStart = configuration.shouldCacheAdditionalAdAtStart
        requestConfiguration = configuration.requestConfiguration
        minDelayMillis = configuration.minDelayMillis
    }

    /**
     * Defines if the cache should load additional ad at the beginning.
     *
     * @param shouldCacheAdditionalAdAtStart Boolean indicating if additional ad should be cached.
     */
    fun setShouldCacheAdditionalAdAtStart(shouldCacheAdditionalAdAtStart: Boolean) {
        this.shouldCacheAdditionalAdAtStart = shouldCacheAdditionalAdAtStart
    }

    /**
     * Sets the minimum delay between two banner consumptions. 1s by default.
     *
     * @param minDelay The minimum delay in seconds between two banner consumptions.
     */
    fun setMinimumDelay(minDelay: Int) {
        minDelayMillis = minDelay * 1000
    }

    fun shouldCacheAdditionalAdAtStart(): Boolean {
        return shouldCacheAdditionalAdAtStart
    }

    override fun toString(): String {
        return "BannerCacheConfiguration{" +
                "placementName='" + placementName + '\'' +
                ", size=" + size +
                ", delegate=" + delegate +
                ", shouldCacheAdditionalAdAtStart=" + shouldCacheAdditionalAdAtStart +
                ", requestConfiguration=" + requestConfiguration +
                ", minDelayMillis=" + minDelayMillis +
                '}'
    }

    companion object {
        private const val MAX_CACHE_SIZE = 5
    }
}