package com.intentsoftware.addapptr.consent

import android.app.Activity
import com.intentsoftware.addapptr.ManagedConsent
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.sfbx.appconsentv3.AppConsent
import com.sfbx.appconsentv3.ui.AppConsentSDK
import com.sfbx.appconsentv3.ui.listener.OnPresentNoticeListener
import com.sfbx.appconsentv3.ui.model.ACConfiguration
import com.sfbx.appconsentv3.ui.model.ACConsentStatus

/**
 * Wrapper for SFBX CMP
 * @param activity Your Activity instance
 */
class CMPAppConsent(
    activity: Activity,
    private val appKey: String
) : CMPImplementation() {

    private var delegate: CMPDelegate? = null

    private var appConsent: AppConsent? = null

    override fun showIfNeeded(activity: Activity) {
        tryToDisplayCmpAndCheckUpdateIfNeeded()
    }

    override fun editConsent(activity: Activity) {
        if (appConsent != null) {
            setOnPresentNoticeListener()
            if (appConsent?.tryToDisplayNotice(force = true) == true) {
                delegate?.didShowCMP()
            }
        } else {
            delegate?.onCMPFailedToShow("Consent form not ready")
        }
    }

    override fun setDelegate(delegate: CMPDelegate) {
        this.delegate = delegate
    }

    override fun reload(activity: Activity) {
    }

    private fun initCmpModule() {
        if (!AppConsentSDK.isSdkInitialized()) {
            val acConfiguration = ACConfiguration.Builder()
                .setForceApplyGDPR(true)
                .setFullScreenMode(true)
                .setNeedToDisplayValidationButtonsVertically(isNeedToDisplayButtonsAtVertical = true)
                .build()

            AppConsentSDK.initialize(
                appKey = appKey,
                configuration = acConfiguration
            ) { appConsentInitialized ->
                appConsent = appConsentInitialized
                setOnPresentNoticeListener()
                delegate?.CMPNeedsUI()
            }
        } else {
            if (appConsent == null) {
                appConsent = AppConsentSDK.getInstance()
            }
        }
    }

    private fun removeCMPCallback() {
        appConsent?.setOnPresentNoticeListener(null)
    }

    private fun tryToDisplayCMP(): Boolean {
        val shown = appConsent?.tryToDisplayNotice(false) ?: false
        if (shown) {
            delegate?.didShowCMP()
        }
        return shown
    }

    private fun setOnPresentNoticeListener() {
        appConsent?.setOnPresentNoticeListener(object : OnPresentNoticeListener {
            override fun presentConsentError(error: Throwable?) {
                delegate?.onCMPFailedToShow(error?.localizedMessage ?: "Failed to present")
                removeCMPCallback()
            }

            override fun presentConsentGiven() {
                val state: ManagedConsent.ManagedConsentState = if (appConsent?.userAcceptAll() == true) {
                    ManagedConsent.ManagedConsentState.OBTAINED
                } else {
                    val obtained = appConsent?.getAllConsentables()?.filter { it.status == ACConsentStatus.ALLOWED}
                    if (obtained?.isNotEmpty() == true) {
                        ManagedConsent.ManagedConsentState.CUSTOM
                    } else {
                        ManagedConsent.ManagedConsentState.WITHHELD
                    }
                }

                delegate?.onConsentUpdated(state)
                removeCMPCallback()
            }
        })
    }

    private fun tryToDisplayCmpAndCheckUpdateIfNeeded() {
        if (!tryToDisplayCMP()) {
            checkIfNoticeHasBeenUpdated()
        }
    }

    private fun checkIfNoticeHasBeenUpdated() {
        appConsent?.checkForUpdate({ isNeedToPresentTheCmp: Boolean ->
            if (isNeedToPresentTheCmp) {
                appConsent?.clearConsent()
                tryToDisplayCMP()
            }
        }) { _: Throwable? ->

        }
    }

    init {
        if (checkRequiredClasses(
                "com.sfbx.appconsentv3.AppConsent",
                "com.sfbx.appconsentv3.ui.AppConsentSDK",
                "com.sfbx.appconsentv3.ui.listener.OnPresentNoticeListener",
                "com.sfbx.appconsentv3.ui.model.ACConfiguration"
            )
        ) {
            initCmpModule()
            onSuccessfulInitialization()
        }
    }
}