package com.intentsoftware.addapptr

import com.intentsoftware.addapptr.internal.module.PlacementHistoryInfo

interface BannerCache {
    /**
     * Sets the [impression listener][ImpressionListener] for banner cache.
     */
    var impressionListener: ImpressionListener?

    /**
     * Sets the [statistics listener][AATKit.StatisticsListener] that will be notified about placement reporting events, like counted adspace, request etc.
     */
    var statisticsListener: AATKit.StatisticsListener?

    /**
     * Setting for Google collapsible banners. Default is null, meaning disabled support for collapsible banners.
     */
    var collapsibleBannerOptions: CollapsibleBannerOptions?

    /**
     * Delegate representing the cache status.
     */
    var cacheStatusDelegate: CacheStatusDelegate?

    /**
     * Updates the configuration that will be used when requesting new banners.
     *
     * @param requestConfiguration [BannerRequest] instance. Can not be null.
     * @param shouldRefresh        True if the whole cache should be re-loaded with new banner request configuration, false if new configuration should only be used for new requests.
     */
    fun updateRequestConfiguration(requestConfiguration: BannerRequest?, shouldRefresh: Boolean)

    /**
     * Returns an instance of [BannerPlacementLayout] to be used within the app. Also automatically counts an ad space.
     *
     *
     * **BannerCache will no longer hold any references to returned banners, and they need to be destroyed manually by the app.**
     *
     *
     * This method respects the frequency capping, set by [BannerCacheConfiguration.setMinimumDelay]
     *
     * @return [BannerPlacementLayout] instance.
     */
    fun consume(): BannerPlacementLayout?

    /**
     * Returns an instance of [BannerPlacementLayout] to be used within the app. Also automatically counts an ad space.
     * **BannerCache will no longer hold any references to returned banners, and they need to be destroyed manually by the app.**
     *
     * @param force true if cache should try to return banner ignoring the frequency capping set by [BannerCacheConfiguration.setMinimumDelay]
     * @return [BannerPlacementLayout] instance.
     */
    fun consume(force: Boolean): BannerPlacementLayout?

    /**
     * Enables the banner creative history.
     */
    fun enableCreativeHistory(size: Int)

    /**
     * Retrieves the banner creative history (if it was enabled).
     */
    fun getCreativeHistory(): Array<PlacementHistoryInfo>?

    /**
     * Destroys the BannerCache, clearing all preloaded banner ads and canceling pending reload requests. For proper memory management, it needs to be called when the BannerCache is no longer needed.
     * **Destroyed BannerCache can no longer be used.**
     */
    fun destroy()

    /**
     * Optional delegate informing about events in BannerCache.
     */
    interface CacheDelegate {
        /**
         * Called when the first banner gets loaded for the cache. Only called once.
         */
        fun firstBannerLoaded()
    }

    /**
     * Optional delegate representing the cache status.
     */
    interface CacheStatusDelegate {
        /**
         * Will be called when the cache was empty and has been filled with at least one ad.
         */
        fun cacheIsNoLongerEmpty()

        /**
         * Will be called when the las ad from cache has been consumed and cache became (temporarily) empty.
         */
        fun cacheIsEmpty()

    }
}