package com.intentsoftware.addapptr.consent

import android.app.Activity
import android.content.Context
import com.google.android.ump.ConsentForm
import com.google.android.ump.ConsentInformation
import com.google.android.ump.ConsentRequestParameters
import com.google.android.ump.FormError
import com.google.android.ump.UserMessagingPlatform
import com.intentsoftware.addapptr.ManagedConsent.ManagedConsentState
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.v
import com.intentsoftware.addapptr.internal.module.Logger.w

/**
 * Wrapper for Google CMP
 * @param activity Your Activity instance
 */
class CMPGoogle(activity: Activity) : CMPImplementation() {
    private enum class MissedCallback {
        FAILED_TO_LOAD,
        FAILED_TO_SHOW,
        NEEDS_UI,
        CONSENT_UPDATED
    }

    private var delegate: CMPDelegate? = null
    private var consentInformation: ConsentInformation? = null
    private var consentForm: ConsentForm? = null
    private var needsUI = false
    private var consentInfoLoading = false
    private var consentFormLoading = false
    private var missedCallback //to handle Google notifications sent before delegate is set
            : MissedCallback? = null
    private var missedCallbackErrorDescription: String? = null

    public override fun setDelegate(delegate: CMPDelegate) {
        this.delegate = delegate
        missedCallback?.let {
            handleMissedCallback(it)
            missedCallback = null
        }
    }

    private fun handleMissedCallback(missedCallback: MissedCallback) {
        when (missedCallback) {
            MissedCallback.FAILED_TO_LOAD -> delegate?.onCMPFailedToLoad(
                missedCallbackErrorDescription!!
            )

            MissedCallback.FAILED_TO_SHOW -> delegate?.onCMPFailedToShow(
                missedCallbackErrorDescription!!
            )

            MissedCallback.NEEDS_UI -> delegate?.CMPNeedsUI()
            MissedCallback.CONSENT_UPDATED -> delegate?.onConsentUpdated(ManagedConsentState.UNKNOWN) //this should actually never happen, so we are good just leaving it as UNKNOWN
        }
        missedCallbackErrorDescription = null
    }

    private fun loadConsentInfo(activity: Activity) {
        if (consentInfoLoading) {
            v(this, { "Consent info already loading, reload request ignored" })
            return
        }
        v(this, { "Loading consent info" })
        val params = ConsentRequestParameters.Builder().build()
        consentInfoLoading = true
        val consentInformation = UserMessagingPlatform.getConsentInformation(activity)
        consentInformation.requestConsentInfoUpdate(activity, params,
            {
                v(this@CMPGoogle, { "Consent info update successful" })
                consentInfoLoading = false
                if (consentInformation.isConsentFormAvailable) {
                    this@CMPGoogle.consentInformation = consentInformation
                    loadForm(activity)
                } else {
                    w(this@CMPGoogle, { "Consent form not available" })
                    val description = "Consent form not available"
                    if (delegate != null) {
                        delegate!!.onCMPFailedToLoad(description)
                    } else {
                        missedCallback = MissedCallback.FAILED_TO_LOAD
                        missedCallbackErrorDescription = description
                    }
                }
            }
        ) { formError: FormError ->
            consentInfoLoading = false
            val description = "Failed to update consent info: ${formError.message}"

            if (delegate != null) {
                delegate!!.onCMPFailedToLoad(description)
            } else {
                missedCallback = MissedCallback.FAILED_TO_LOAD
                missedCallbackErrorDescription = description
            }
        }
    }

    private fun loadForm(context: Context) {
        if (consentFormLoading) {
            v(this, { "Consent form already loading, reload request ignored" })
            return
        }
        v(this, { "Loading consent form" })
        consentFormLoading = true
        UserMessagingPlatform.loadConsentForm(context,
            { consentForm: ConsentForm? ->
                v(this@CMPGoogle, { "Consent form loaded" })
                consentFormLoading = false
                this@CMPGoogle.consentForm = consentForm
                if (consentInformation?.consentStatus == ConsentInformation.ConsentStatus.REQUIRED) {
                    needsUI = true
                }
                if (needsUI) { //handle also situations when form loads after user already tried to show it
                    if (delegate != null) {
                        delegate!!.CMPNeedsUI()
                    } else {
                        missedCallback = MissedCallback.NEEDS_UI
                    }
                }
            }
        ) { formError: FormError ->
            consentFormLoading = false
            val description = "Failed to load consent form: " + formError.message
            if (delegate != null) {
                delegate!!.onCMPFailedToLoad(description)
            } else {
                missedCallback = MissedCallback.FAILED_TO_LOAD
                missedCallbackErrorDescription = description
            }
        }
    }

    public override fun reload(activity: Activity) {
        if (consentInformation == null) {
            loadConsentInfo(activity)
        } else if (consentForm == null) {
            loadForm(activity)
        }
    }

    override fun showIfNeeded(activity: Activity) {
        if (needsUI) {
            editConsent(activity)
        }
    }

    public override fun editConsent(activity: Activity) {
        needsUI = true
        if (consentForm != null) {
            needsUI = false
            delegate?.didShowCMP()
            consentForm!!.show(
                activity
            ) {
                // Handle dismissal by reloading form.
                consentForm = null
                loadForm(activity)
                if (delegate != null) {
                    val state: ManagedConsentState
                    val vendorsConsentString =
                        activity.getSharedPreferences(activity.packageName + "_preferences", Context.MODE_PRIVATE)
                            .getString("IABTCF_VendorConsents", null)
                    val purposesConsentString =
                        activity.getSharedPreferences(activity.packageName + "_preferences", Context.MODE_PRIVATE)
                            .getString("IABTCF_PurposeConsents", null)
                    state = if (vendorsConsentString == null || purposesConsentString == null) {
                        e(this, {
                            """
                                 At least one of the required consent strings was not saved!
                                 vendor consents: $vendorsConsentString
                                 purpose consents:$purposesConsentString
                                 """.trimIndent()
                        })

                        ManagedConsentState.UNKNOWN
                    } else if (vendorsConsentString.contains("1") && purposesConsentString.contains(
                            "1"
                        )
                    ) { //at least one vendor and one purpose enabled
                        ManagedConsentState.CUSTOM
                    } else {
                        ManagedConsentState.WITHHELD
                    }
                    delegate!!.onConsentUpdated(state)
                } else {
                    missedCallback = MissedCallback.CONSENT_UPDATED
                }
            }
        } else {
            val description = "Consent form not ready"
            if (delegate != null) {
                delegate!!.onCMPFailedToShow(description)
            } else {
                missedCallback = MissedCallback.FAILED_TO_SHOW
                missedCallbackErrorDescription = description
            }
        }
    }

    init {
        if (checkRequiredClasses(
                "com.google.android.ump.ConsentForm",
                "com.google.android.ump.ConsentInformation",
                "com.google.android.ump.ConsentRequestParameters",
                "com.google.android.ump.FormError",
                "com.google.android.ump.UserMessagingPlatform"
            )
        ) {
            loadConsentInfo(activity)
            onSuccessfulInitialization()
        }
    }
}