package com.intentsoftware.addapptr.consent

import android.app.Activity
import android.view.View
import com.intentsoftware.addapptr.ManagedConsent
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.sourcepoint.cmplibrary.NativeMessageController
import com.sourcepoint.cmplibrary.SpClient
import com.sourcepoint.cmplibrary.core.nativemessage.MessageStructure
import com.sourcepoint.cmplibrary.creation.config
import com.sourcepoint.cmplibrary.creation.delegate.spConsentLibLazy
import com.sourcepoint.cmplibrary.data.network.util.CampaignType
import com.sourcepoint.cmplibrary.model.ConsentAction
import com.sourcepoint.cmplibrary.model.MessageLanguage
import com.sourcepoint.cmplibrary.model.PMTab
import com.sourcepoint.cmplibrary.model.exposed.SPConsents
import com.sourcepoint.cmplibrary.model.exposed.SpConfig
import org.json.JSONObject


class CMPSourcePoint(
    activity: Activity,
    accountId: Int,
    propertyId: Int,
    propertyName: String,
    private val pmId: String
) : CMPImplementation() {

    private var delegate: CMPDelegate? = null
    private var tryingToShowCMP = false
    private var cmpUI: View? = null

    private val cmpConfig: SpConfig = config {
        this.accountId = accountId
        this.propertyId = propertyId
        this.propertyName = propertyName
        messLanguage = MessageLanguage.ENGLISH // Optional, default ENGLISH
        +CampaignType.GDPR
    }

    private val spConsentLib by spConsentLibLazy {
        this.activity = activity
        spClient = LocalClient()
        spConfig = cmpConfig
    }

    override fun showIfNeeded(activity: Activity) {
        cmpUI?.let {
            tryingToShowCMP = true
            spConsentLib.showView(it)
        }
    }

    override fun editConsent(activity: Activity) {
        tryingToShowCMP = true
        spConsentLib.loadPrivacyManager(pmId, PMTab.DEFAULT, CampaignType.GDPR)
    }

    override fun setDelegate(delegate: CMPDelegate) {
        this.delegate = delegate
    }

    override fun reload(activity: Activity) {
    }

    internal inner class LocalClient : SpClient {
        override fun onUIFinished(view: View) {
            tryingToShowCMP = false
            spConsentLib.removeView(view)
        }

        override fun onUIReady(view: View) {
            cmpUI = view
            if (tryingToShowCMP) {
                spConsentLib.showView(view)
            } else {
                delegate?.CMPNeedsUI()
            }
        }

        @Deprecated("onMessageReady callback will be removed in favor of onUIReady. Currently this callback is disabled.")
        override fun onMessageReady(message: JSONObject) {} // Deprecated
        override fun onNativeMessageReady(
            message: MessageStructure,
            messageController: NativeMessageController
        ) {
        }

        override fun onError(error: Throwable) {
            if (tryingToShowCMP)
                delegate?.onCMPFailedToShow(error.toString())
            else
                delegate?.onCMPFailedToLoad(error.toString())
        }

        override fun onConsentReady(consent: SPConsents) {

            if (consent.gdpr?.consent?.grants != null) {
                val numberOfObtained =
                    consent.gdpr?.consent?.grants?.filter { it.value.granted } ?: emptyMap()
                val numberOfWithheld =
                    consent.gdpr?.consent?.grants?.filter { !it.value.granted } ?: emptyMap()

                delegate?.onConsentUpdated(
                    when {
                        numberOfObtained.isNotEmpty() && numberOfWithheld.isNotEmpty() -> ManagedConsent.ManagedConsentState.CUSTOM
                        numberOfObtained.isNotEmpty() -> ManagedConsent.ManagedConsentState.OBTAINED
                        numberOfWithheld.isNotEmpty() -> ManagedConsent.ManagedConsentState.WITHHELD
                        else -> ManagedConsent.ManagedConsentState.UNKNOWN
                    }
                )
            } else {
                delegate?.onConsentUpdated(ManagedConsent.ManagedConsentState.UNKNOWN)
            }
        }

        override fun onAction(view: View, consentAction: ConsentAction): ConsentAction {
            delegate?.didShowCMP()
            return consentAction
        }

        override fun onNoIntentActivitiesFound(url: String) {}
        override fun onSpFinished(sPConsents: SPConsents) {
        }
    }

    companion object {
        private const val MORE_PARTNERS_STRING_KEY = "IABTCF_AddtlConsent"
    }

    init {
        if (checkRequiredClasses(
                "com.sourcepoint.cmplibrary.SpClient",
                "com.sourcepoint.cmplibrary.NativeMessageController",
                "com.sourcepoint.cmplibrary.model.exposed.SpConfig"
            )
        ) {
            onSuccessfulInitialization()
            spConsentLib.loadMessage()
        }
    }
}