package com.intentsoftware.addapptr.consent

import android.app.Activity
import android.content.Context
import android.util.Log
import android.view.View
import com.intentsoftware.addapptr.AdNetwork
import com.intentsoftware.addapptr.ManagedConsent
import com.intentsoftware.addapptr.NonIABConsent
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.intentsoftware.addapptr.internal.module.Logger
import com.sourcepoint.cmplibrary.NativeMessageController
import com.sourcepoint.cmplibrary.SpClient
import com.sourcepoint.cmplibrary.core.nativemessage.MessageStructure
import com.sourcepoint.cmplibrary.creation.config
import com.sourcepoint.cmplibrary.creation.delegate.spConsentLibLazy
import com.sourcepoint.cmplibrary.exception.CampaignType
import com.sourcepoint.cmplibrary.model.ConsentAction
import com.sourcepoint.cmplibrary.model.MessageLanguage
import com.sourcepoint.cmplibrary.model.PMTab
import com.sourcepoint.cmplibrary.model.exposed.SPConsents
import com.sourcepoint.cmplibrary.model.exposed.SpConfig
import org.json.JSONObject


class CMPSourcePoint(
    activity: Activity,
    accountId: Int,
    propertyId: Int,
    propertyName: String,
    private val pmId: String
) : CMPImplementation() {

    private var delegate: CMPDelegate? = null
    private var tryingToShowCMP = false
    private var cmpUI: View? = null
    private var enabledAdditionalPartners: List<String>? = null

    private val cmpConfig: SpConfig = config {
        this.accountId = accountId
        this.propertyId = propertyId
        this.propertyName = propertyName
        messLanguage = MessageLanguage.ENGLISH // Optional, default ENGLISH
        +CampaignType.GDPR
    }

    private val spConsentLib by spConsentLibLazy {
        this.activity = activity
        spClient = LocalClient()
        spConfig = cmpConfig
    }

    override fun showIfNeeded(activity: Activity) {
        cmpUI?.let {
            tryingToShowCMP = true
            spConsentLib.showView(it)
        }
    }

    override fun editConsent(activity: Activity) {
        tryingToShowCMP = true
        spConsentLib.loadPrivacyManager(pmId, PMTab.DEFAULT, CampaignType.GDPR)
    }

    override fun setDelegate(delegate: CMPDelegate) {
        this.delegate = delegate
    }

    override fun reload(activity: Activity) {
    }

    // see https://storage.googleapis.com/tcfac/additional-consent-providers.csv for partners list
    override fun getConsentForNetwork(network: AdNetwork): NonIABConsent {
        return when (network) {
            AdNetwork.APPLOVIN, AdNetwork.APPLOVINMAX -> readConsentForProviderId(1301)
            AdNetwork.FACEBOOK -> readConsentForProviderId(89)
            AdNetwork.PUBNATIVE -> readConsentForProviderId(2977)
            AdNetwork.IRONSOURCE -> readConsentForProviderId(2878)
            AdNetwork.UNITY -> readConsentForProviderId(3234)
            AdNetwork.VUNGLE2 -> readConsentForProviderId(2707)
            AdNetwork.KIDOZ -> readConsentForProviderId(3183)
            else -> {
                //FIXME: missing mappings for Huawei
                if (Logger.isLoggable(Log.WARN)) {
                    Logger.w(
                        this,
                        "No mapping for network $network available, treating consent as withheld"
                    )
                }
                NonIABConsent.WITHHELD
            }
        }
    }

    private fun updateMorePartnersData(activity: Activity) {
        var morePartnersConsentString =
            activity.getSharedPreferences(
                activity.packageName + "_preferences",
                Context.MODE_PRIVATE
            ).getString(
                MORE_PARTNERS_STRING_KEY, null
            )

        enabledAdditionalPartners = null
        if (morePartnersConsentString != null) {
            morePartnersConsentString =
                morePartnersConsentString.replaceFirst(".*~".toRegex(), "") //remove "1~" or similar
            val partnersArray = morePartnersConsentString.split("\\.".toRegex()).toTypedArray()
            enabledAdditionalPartners = listOf(*partnersArray)
        } else {
            if (Logger.isLoggable(Log.VERBOSE)) {
                Logger.v(this, "No more partners string found")
            }
        }
    }

    private fun readConsentForProviderId(id: Int): NonIABConsent {
        return if (enabledAdditionalPartners != null) {
            if (enabledAdditionalPartners!!.contains(id.toString())) {
                NonIABConsent.OBTAINED
            } else {
                NonIABConsent.WITHHELD
            }
        } else {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(
                    this,
                    "Enabled additional partners list is not available"
                )
            }
            NonIABConsent.WITHHELD
        }
    }

    internal inner class LocalClient : SpClient {
        override fun onUIFinished(view: View) {
            tryingToShowCMP = false
            spConsentLib.removeView(view)
        }

        override fun onUIReady(view: View) {
            cmpUI = view
            if (tryingToShowCMP) {
                spConsentLib.showView(view)
            } else {
                delegate?.CMPNeedsUI()
            }
        }

        override fun onMessageReady(message: JSONObject) {} // Deprecated
        override fun onNativeMessageReady(
            message: MessageStructure,
            messageController: NativeMessageController
        ) {
        }

        override fun onError(error: Throwable) {
            if (tryingToShowCMP)
                delegate?.onCMPFailedToShow(error.toString())
            else
                delegate?.onCMPFailedToLoad(error.toString())
        }

        override fun onConsentReady(consent: SPConsents) {
            if (consent.gdpr?.consent?.grants != null) {
                val numberOfObtained =
                    consent.gdpr?.consent?.grants?.filter { it.value.granted } ?: emptyMap()
                val numberOfWithheld =
                    consent.gdpr?.consent?.grants?.filter { !it.value.granted } ?: emptyMap()

                delegate?.onConsentUpdated(
                    when {
                        numberOfObtained.isNotEmpty() && numberOfWithheld.isNotEmpty() -> ManagedConsent.ManagedConsentState.CUSTOM
                        numberOfObtained.isNotEmpty() -> ManagedConsent.ManagedConsentState.OBTAINED
                        numberOfWithheld.isNotEmpty() -> ManagedConsent.ManagedConsentState.WITHHELD
                        else -> ManagedConsent.ManagedConsentState.UNKNOWN
                    }
                )
            } else {
                delegate?.onConsentUpdated(ManagedConsent.ManagedConsentState.UNKNOWN)
            }
        }

        override fun onAction(view: View, consentAction: ConsentAction): ConsentAction =
            consentAction

        override fun onNoIntentActivitiesFound(url: String) {}
        override fun onSpFinished(sPConsents: SPConsents) {
        }
    }

    companion object {
        private const val MORE_PARTNERS_STRING_KEY = "IABTCF_AddtlConsent"
    }

    init {
        if (checkRequiredClasses(
                "com.sourcepoint.cmplibrary.SpClient",
                "com.sourcepoint.cmplibrary.NativeMessageController",
                "com.sourcepoint.cmplibrary.model.exposed.SpConfig"
            )
        ) {
            onSuccessfulInitialization()
            updateMorePartnersData(activity)
            spConsentLib.loadMessage()
        }
    }
}