package com.intentsoftware.addapptr.consent

import android.app.Activity
import android.content.Context
import com.google.android.ump.*
import com.intentsoftware.addapptr.AdNetwork
import com.intentsoftware.addapptr.ManagedConsent.ManagedConsentState
import com.intentsoftware.addapptr.NonIABConsent
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.v
import com.intentsoftware.addapptr.internal.module.Logger.w

/**
 * Wrapper for Google CMP
 * @param activity Your Activity instance
 */
class CMPGoogle(activity: Activity) : CMPImplementation() {
    private enum class MissedCallback {
        FAILED_TO_LOAD,
        FAILED_TO_SHOW,
        NEEDS_UI,
        CONSENT_UPDATED
    }

    private var delegate: CMPDelegate? = null
    private var consentInformation: ConsentInformation? = null
    private var consentForm: ConsentForm? = null
    private var needsUI = false
    private var enabledAdditionalPartners: List<String>? = null
    private var consentInfoLoading = false
    private var consentFormLoading = false
    private var missedCallback //to handle Google notifications sent before delegate is set
            : MissedCallback? = null
    private var missedCallbackErrorDescription: String? = null

    public override fun setDelegate(delegate: CMPDelegate) {
        this.delegate = delegate
        missedCallback?.let {
            handleMissedCallback(it)
            missedCallback = null
        }
    }

    private fun handleMissedCallback(missedCallback: MissedCallback) {
        when (missedCallback) {
            MissedCallback.FAILED_TO_LOAD -> delegate?.onCMPFailedToLoad(
                missedCallbackErrorDescription!!
            )

            MissedCallback.FAILED_TO_SHOW -> delegate?.onCMPFailedToShow(
                missedCallbackErrorDescription!!
            )

            MissedCallback.NEEDS_UI -> delegate?.CMPNeedsUI()
            MissedCallback.CONSENT_UPDATED -> delegate?.onConsentUpdated(ManagedConsentState.UNKNOWN) //this should actually never happen, so we are good just leaving it as UNKNOWN
        }
        missedCallbackErrorDescription = null
    }

    private fun loadConsentInfo(activity: Activity) {
        if (consentInfoLoading) {
            v(this, { "Consent info already loading, reload request ignored" })
            return
        }
        v(this, { "Loading consent info" })
        val params = ConsentRequestParameters.Builder().build()
        consentInfoLoading = true
        val consentInformation = UserMessagingPlatform.getConsentInformation(activity)
        consentInformation.requestConsentInfoUpdate(activity, params,
            {
                v(this@CMPGoogle, { "Consent info update successful" })
                consentInfoLoading = false
                if (consentInformation.isConsentFormAvailable) {
                    this@CMPGoogle.consentInformation = consentInformation
                    loadForm(activity)
                } else {
                    w(this@CMPGoogle, { "Consent form not available" })
                    val description = "Consent form not available"
                    if (delegate != null) {
                        delegate!!.onCMPFailedToLoad(description)
                    } else {
                        missedCallback = MissedCallback.FAILED_TO_LOAD
                        missedCallbackErrorDescription = description
                    }
                }
            }
        ) { formError: FormError ->
            consentInfoLoading = false
            val description = "Failed to update consent info: ${formError.message}"

            if (delegate != null) {
                delegate!!.onCMPFailedToLoad(description)
            } else {
                missedCallback = MissedCallback.FAILED_TO_LOAD
                missedCallbackErrorDescription = description
            }
        }
    }

    private fun loadForm(context: Context) {
        if (consentFormLoading) {
            v(this, { "Consent form already loading, reload request ignored" })
            return
        }
        v(this, { "Loading consent form" })
        consentFormLoading = true
        UserMessagingPlatform.loadConsentForm(context,
            { consentForm: ConsentForm? ->
                v(this@CMPGoogle, { "Consent form loaded" })
                consentFormLoading = false
                this@CMPGoogle.consentForm = consentForm
                if (consentInformation?.consentStatus == ConsentInformation.ConsentStatus.REQUIRED) {
                    needsUI = true
                    if (delegate != null) {
                        delegate!!.CMPNeedsUI()
                    } else {
                        missedCallback = MissedCallback.NEEDS_UI
                    }
                }
            }
        ) { formError: FormError ->
            consentFormLoading = false
            val description = "Failed to load consent form: " + formError.message
            if (delegate != null) {
                delegate!!.onCMPFailedToLoad(description)
            } else {
                missedCallback = MissedCallback.FAILED_TO_LOAD
                missedCallbackErrorDescription = description
            }
        }
    }

    public override fun reload(activity: Activity) {
        if (consentInformation == null) {
            loadConsentInfo(activity)
        } else if (consentForm == null) {
            loadForm(activity)
        }
    }

    override fun showIfNeeded(activity: Activity) {
        if (needsUI) {
            editConsent(activity)
        }
    }

    public override fun editConsent(activity: Activity) {
        if (consentForm != null) {
            needsUI = false
            consentForm!!.show(
                activity
            ) {
                // Handle dismissal by reloading form.
                updateMorePartnersData(activity)
                consentForm = null
                loadForm(activity)
                if (delegate != null) {
                    val state: ManagedConsentState
                    val vendorsConsentString =
                        activity.getSharedPreferences(activity.packageName + "_preferences", Context.MODE_PRIVATE)
                            .getString("IABTCF_VendorConsents", null)
                    val purposesConsentString =
                        activity.getSharedPreferences(activity.packageName + "_preferences", Context.MODE_PRIVATE)
                            .getString("IABTCF_PurposeConsents", null)
                    state = if (vendorsConsentString == null || purposesConsentString == null) {
                        e(this, {
                            """
                                 At least one of the required consent strings was not saved!
                                 vendor consents: $vendorsConsentString
                                 purpose consents:$purposesConsentString
                                 """.trimIndent()
                        })

                        ManagedConsentState.UNKNOWN
                    } else if (vendorsConsentString.contains("1") && purposesConsentString.contains(
                            "1"
                        )
                    ) { //at least one vendor and one purpose enabled
                        ManagedConsentState.CUSTOM
                    } else {
                        ManagedConsentState.WITHHELD
                    }
                    delegate!!.onConsentUpdated(state)
                } else {
                    missedCallback = MissedCallback.CONSENT_UPDATED
                }
            }
        } else {
            val description = "Consent form not ready"
            if (delegate != null) {
                delegate!!.onCMPFailedToShow(description)
            } else {
                missedCallback = MissedCallback.FAILED_TO_SHOW
                missedCallbackErrorDescription = description
            }
        }
    }

    // see https://storage.googleapis.com/tcfac/additional-consent-providers.csv for partners list
    public override fun getConsentForNetwork(network: AdNetwork): NonIABConsent {
        return when (network) {
            AdNetwork.APPLOVIN, AdNetwork.APPLOVINMAX -> readConsentForProviderId(1301)
            AdNetwork.FACEBOOK -> readConsentForProviderId(89)
            AdNetwork.PUBNATIVE -> readConsentForProviderId(2977)
            AdNetwork.IRONSOURCENEW -> readConsentForProviderId(2878)
            AdNetwork.UNITY -> readConsentForProviderId(3234)
            AdNetwork.VUNGLE2 -> readConsentForProviderId(2707)
            AdNetwork.KIDOZ -> readConsentForProviderId(3183)
            else -> {
                //FIXME: missing mappings for Huawei, InMobi
                w(this, { "No mapping for network $network available, treating consent as withheld" })
                NonIABConsent.WITHHELD
            }
        }
    }

    private fun updateMorePartnersData(activity: Activity) {
        var morePartnersConsentString =
            activity.getSharedPreferences(activity.packageName + "_preferences", Context.MODE_PRIVATE).getString(MORE_PARTNERS_STRING_KEY, null)

        enabledAdditionalPartners = null
        if (morePartnersConsentString != null) {
            morePartnersConsentString =
                morePartnersConsentString.replaceFirst(".*~".toRegex(), "") //remove "1~" or similar
            val partnersArray = morePartnersConsentString.split("\\.".toRegex()).toTypedArray()
            enabledAdditionalPartners = listOf(*partnersArray)
        } else {
            v(this, { "No more partners string found" })
        }
    }

    private fun readConsentForProviderId(id: Int): NonIABConsent {
        return if (enabledAdditionalPartners != null) {
            if (enabledAdditionalPartners!!.contains(id.toString())) {
                NonIABConsent.OBTAINED
            } else {
                NonIABConsent.WITHHELD
            }
        } else {
            w(this, { "Enabled additional partners list is not available" })
            NonIABConsent.WITHHELD
        }
    }

    companion object {
        private const val MORE_PARTNERS_STRING_KEY = "IABTCF_AddtlConsent"
    }

    init {
        if (checkRequiredClasses(
                "com.google.android.ump.ConsentForm",
                "com.google.android.ump.ConsentInformation",
                "com.google.android.ump.ConsentRequestParameters",
                "com.google.android.ump.FormError",
                "com.google.android.ump.UserMessagingPlatform"
            )
        ) {
            updateMorePartnersData(activity)
            loadConsentInfo(activity)
            onSuccessfulInitialization()
        }
    }
}