package com.intentsoftware.addapptr;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.util.Log;
import android.util.Pair;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;

import com.intentsoftware.addapptr.ad.NativeAd;
import com.intentsoftware.addapptr.ad.NativeAdData;
import com.intentsoftware.addapptr.ad.VASTAdData;
import com.intentsoftware.addapptr.module.Logger;
import com.intentsoftware.addapptr.module.ServerLogger;
import com.intentsoftware.addapptr.module.Utils;

import java.util.List;
import java.util.Map;
import java.util.Set;

import androidx.annotation.Nullable;

@SuppressWarnings({"unused", "WeakerAccess"})
public class AATKit {

    // Used only in Javadoc.
    public static final int BANNER_DEFAULT_RELOAD_INTERVAL_IN_SECONDS = (int) (BannerReloader.AUTORELOAD_INTERVAL_DEFAULT / 1000);
    public static final int BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS = (int) (BannerReloader.AUTORELOAD_INTERVAL_MIN / 1000);
    public static final int BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS = (int) (BannerReloader.AUTORELOAD_INTERVAL_MAX / 1000);

    /**
     * Used for passing of detailed GDPR consent. Possible implementations: {@link ManagedConsent}, {@link SimpleConsent}, {@link VendorConsent}.
     */
    public interface Consent {
        /**
         * Sets to set the networks that will not be requested if they do not have the TCF2 consent. Only works if TCF2 compliant CMP is used.
         *
         * @param stopSet Set of networks not to be requested without TCF2 consent.
         */
        void setNoConsentNetworkStopSet(Set<AdNetwork> stopSet);
    }

    /**
     * Notifies about AATKit events.
     */
    public interface Delegate {
        /**
         * Notifies that placement has finished loading an ad successfully.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         */
        void aatkitHaveAd(int placementId);

        /**
         * Notifies that placement has failed to load an ad.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         */
        void aatkitNoAd(int placementId);

        /**
         * Notifies that ad went fullscreen and that application should pause.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         */
        void aatkitPauseForAd(int placementId);

        /**
         * Notifies that ad came back from fullscreen and that application should resume.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         */
        void aatkitResumeAfterAd(int placementId);

        /**
         * Notifies that placement has loaded an Empty ad.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         */
        void aatkitShowingEmpty(int placementId);

        /**
         * Notifies that placement has earned incentive (by rewarded ads).
         * <p>
         * Note: This callback might also be invoked for regular fullscreen
         * placements, not just the ones configured for rewarded videos. Please
         * make sure placementId is your rewarded video placement before rewarding
         * the user.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         * @param aatKitReward Incentive object. Can be null for networks not supporting reward information.
         */
        void aatkitUserEarnedIncentive(int placementId, @Nullable AATKitReward aatKitReward);

        /**
         * Notifies that the AATKit has obtained ad rules.
         *
         * @param fromTheServer Indicates if the rules came from the server. It will return false if the currently used rules come from the {@link #setInitialRules(String)} method or the cached rules are used.
         */
        void aatkitObtainedAdRules(boolean fromTheServer);

        /**
         * Notifies that application's bundle ID was not recognized by the AddApptr server.
         */
        void aatkitUnknownBundleId();

        /**
         * Notifies that AATKit has new banner view ready for placement. Used only for MultiSizeBanner.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         * @param bannerView  Loaded banner view
         */
        void aatkitHaveAdForPlacementWithBannerView(int placementId, BannerPlacementLayout bannerView);

        /**
         * Notifies that placement has finished loading an VAST ad successfully. Used only for VAST.
         *
         * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
         * @param data        Loaded VAST ad data.
         */
        void aatkitHaveVASTAd(int placementId, VASTAdData data);
    }

    private static AdController adController;

    /**
     * @return AATKit version.
     */
    public static String getVersion() {
        logAATKitCall("CMD: getVersion()");
        return Version.NAME;
    }

    /**
     * @return AATKit detailed build version.
     */
    public static String getFullVersion() {
        logAATKitCall("CMD: getFullVersion()");
        return Version.FULL_NAME;
    }

    /**
     * Initializes the AATKit library. Should be called once during application initialization before any other calls to AATKit.
     *
     * @param configuration Configuration for AATKit.
     * @see AATKitConfiguration
     */
    public static void init(AATKitConfiguration configuration) {
        logAATKitCall("CMD: init(" + configuration + ")");

        if (configuration == null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(AATKit.class, "Configuration cannot be null");
            }
            return;
        }
        if (adController != null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(AATKit.class, "AdController is already initialized");
            }
            return;
        }

        String version = "?";
        String packageName = configuration.getApplication().getApplicationContext().getPackageName();
        try {
            version = configuration.getApplication().getPackageManager().getPackageInfo(packageName, 0).versionName;
        } catch (PackageManager.NameNotFoundException e) {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(AATKit.class, "Failed to check version of application", e);
            }
        }

        Logger.d("Init", Version.FULL_NAME
                + ", application: " + packageName
                + " (version: " + version + ")"
                + ", configuration: " + configuration);
        adController = new AdController(configuration);
    }

    /**
     * Initializes the AATKit library. Should be called once during application initialization before any other calls to AATKit.
     *
     * @param application Reference to {@link Application}.
     * @param delegate    Optional delegate allowing to listen for AATKit events.
     * @see AATKit.Delegate
     * @deprecated use {@link #init(AATKitConfiguration)} instead.
     */
    @Deprecated
    public static void init(Application application, Delegate delegate) {
        logAATKitCall("CMD: init(" + application + "," + delegate + ")");
        if (adController != null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(AATKit.class, "AdController is already initialized");
            }
            return;
        }
        Logger.d("Init", Version.FULL_NAME + ", live mode, shake debug on, rule caching on");
        AATKitConfiguration config = new AATKitConfiguration(application);
        config.setDelegate(delegate);
        adController = new AdController(config);

    }

    /**
     * Initializes the AATKit library. Should be called once during application initialization before any other calls to AATKit.
     *
     * @param application       Reference to {@link Application}.
     * @param delegate          Optional delegate allowing to listen for AATKit events.
     * @param enableRuleCaching Sets if AATKit should preserve last downloaded ad rules when the application is closed.
     * @param initialRules      Optional parameter (can be null). Allows to set initial rules that will be used before real config is downloaded from the server.
     * @see AATKit.Delegate
     * @deprecated use {@link #init(AATKitConfiguration)} instead.
     */
    @Deprecated
    public static void init(Application application, Delegate delegate, boolean enableRuleCaching, String initialRules) {
        logAATKitCall("CMD: init(" + application + "," + delegate + "," + enableRuleCaching + "," + initialRules + ")");
        if (adController != null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(AATKit.class, "AdController is already initialized");
            }
            return;
        }
        Logger.d("Init", Version.FULL_NAME + ", live mode, shake debug on, rule caching: " + enableRuleCaching + ", initial rules: " + initialRules);

        AATKitConfiguration config = new AATKitConfiguration(application);
        config.setDelegate(delegate);
        config.setShouldCacheRules(enableRuleCaching);
        config.setInitialRules(initialRules);

        adController = new AdController(config);
    }


    /**
     * Initializes the AATKit library. Should be called once during application initialization before any other calls to AATKit.
     * It also enables the test ads, and should only be used during development.
     *
     * @param application Reference to {@link Application}.
     * @param delegate    Optional delegate allowing to listen for AATKit events.
     * @param testId      Test application id obtained from addapptr.com account.
     * @deprecated use {@link #init(AATKitConfiguration)} instead.
     */
    @Deprecated
    public static void init(Application application, Delegate delegate, int testId) {
        logAATKitCall("CMD: init(" + application + "," + delegate + "," + testId + ")");
        if (adController != null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(AATKit.class, "AdController is already initialized");
            }
            return;
        }
        Logger.d("Init", Version.FULL_NAME + ", test mode with ID: " + testId + ", shake debug on, rule caching on");

        AATKitConfiguration config = new AATKitConfiguration(application);
        config.setDelegate(delegate);
        config.setTestModeAccountId(testId);

        adController = new AdController(config);
    }


    /**
     * Initializes the AATKit library without debug screen appearing after shaking the device (it is enabled by default). It is advised to use the {@link #init(Application, Delegate)} method instead. Should be called once during application initialization before any other calls
     * to AATKit.
     *
     * @param application Reference to {@link Application}.
     * @param delegate    Optional delegate allowing to listen for AATKit events.
     * @see AATKit.Delegate
     * @deprecated use {@link #init(AATKitConfiguration)} instead.
     */
    @Deprecated
    public static void initWithoutDebugScreen(Application application, Delegate delegate) {
        logAATKitCall("CMD: initWithoutDebugScreen(" + application + "," + delegate + ")");
        if (adController != null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(AATKit.class, "AdController is already initialized");
            }
            return;
        }
        Logger.d("Init", Version.FULL_NAME + ", live mode, shake debug off, rule caching on");
        AATKitConfiguration config = new AATKitConfiguration(application);
        config.setDelegate(delegate);
        config.setUseDebugShake(false);

        adController = new AdController(config);
    }

    /**
     * Enables AATKit test ads that should be for testing only during development.
     *
     * @param testAppId Test application id obtained from addapptr.com account.
     * @deprecated use {@link #init(AATKitConfiguration)} with setting the test mode in configuration object instead.
     */
    @Deprecated
    public static void enableTestMode(int testAppId) {
        logAATKitCall("CMD: enableTestMode(" + testAppId + ")");
        adController.setTestAppId(testAppId);
    }

    /**
     * Enables debug screen that will show after shaking the device. It is already enabled by default.
     */
    public static void enableDebugScreen() {
        logAATKitCall("CMD: enableDebugScreen()");
        adController.enableDebugScreen();
    }

    /**
     * Disables the debug screen appearing after shaking the device. It is enabled by default.
     */
    public static void disableDebugScreen() {
        logAATKitCall("CMD: disableDebugScreen()");
        adController.disableDebugScreen();
    }

    /**
     * Used for obtaining debug information (the same that would be presented in dialog after shaking the device if debug screen is enabled)
     *
     * @return String with debug information
     */
    public static String getDebugInfo() {
        logAATKitCall("CMD: getDebugInfo()");
        return adController.getDebugInfo();
    }

    /**
     * Allows to reconfigure the options for GDPR consent.
     *
     * @param configuration New configuration.
     */
    public static void reconfigure(AATKitRuntimeConfiguration configuration) {
        logAATKitCall("CMD: reconfigure(" + configuration + ")");
        adController.reconfigure(configuration);
    }

    /**
     * Checks if AATKit recognizes given device as tablet.
     *
     * @param context The {@link Context} of your application.
     * @return True if device is recognized as tablet, false otherwise.
     */
    public static boolean isTablet(Context context) {
        logAATKitCall("CMD: isTablet(" + context + ")");
        return Utils.isTablet(context);
    }

    /**
     * Returns the {@link PlacementSize} with maximum width that will fit on a given device in portrait screen orientation.
     *
     * @param context The {@link Context} of your application.
     * @return {@link PlacementSize} best fitting current device
     */
    public static PlacementSize maximumBannerSizePortrait(Context context) {
        logAATKitCall("CMD: maximumBannerSizePortrait(" + context + ")");
        return Utils.maxPlacementSize(Utils.screenWidthPortrait(context));
    }

    /**
     * Returns the {@link PlacementSize} with maximum width that will fit on a given device in landscape screen orientation.
     *
     * @param context The {@link Context} of your application.
     * @return {@link PlacementSize} best fitting current device
     */
    public static PlacementSize maximumBannerSizeLandscape(Context context) {
        logAATKitCall("CMD: maximumBannerSizeLandscape(" + context + ")");
        return Utils.maxPlacementSize(Utils.screenWidthLandscape(context));
    }

    /**
     * Returns the set of {@link BannerSize} that will fit on a given device in portrait screen orientation.
     *
     * @param context The {@link Context} of your application.
     * @return Set of {@link BannerSize} fitting current device
     */
    public static Set<BannerSize> fittingBannerSizesPortrait(Context context) {
        logAATKitCall("CMD: fittingBannerSizesPortrait(" + context + ")");
        return Utils.fittingBannerSizes(Utils.screenWidthPortrait(context));
    }

    /**
     * Returns the set of {@link BannerSize} that will fit on a given device in landscape screen orientation.
     *
     * @param context The {@link Context} of your application.
     * @return Set of {@link BannerSize} fitting current device
     */
    public static Set<BannerSize> fittingBannerSizesLandscape(Context context) {
        logAATKitCall("CMD: fittingBannerSizesLandscape(" + context + ")");
        return Utils.fittingBannerSizes(Utils.screenWidthLandscape(context));
    }

    /**
     * Notifies AATKit about activity resume. Invoke this method in every activity that uses AATKit.
     *
     * @param activity Reference to {@link Activity}.
     */
    public static void onActivityResume(Activity activity) {
        logAATKitCall("CMD: onActivityResume(" + activity + ")");
        adController.onActivityResume(activity);
    }

    /**
     * Notifies AATKit about activity pause. Invoke this method in every activity that uses AATKit.
     *
     * @param activity Reference to {@link Activity}.
     */
    public static void onActivityPause(Activity activity) {
        logAATKitCall("CMD: onActivityPause(" + activity + ")");
        adController.onActivityPause();
    }

    /**
     * Creates placement with given name and size. If the placement of given name and size already exists, it will be returned.
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @param size Size of placement. Use {@link PlacementSize#Fullscreen} for interstitials, {@link PlacementSize#Native} for native ads and other sizes for banner ads.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    public static int createPlacement(String name, PlacementSize size) {
        logAATKitCall("CMD: createPlacement(" + name + "," + size + ")");
        return adController.createPlacement(name, size);
    }

    /**
     * Creates a new native ad placement. If the native ad placement of given name already exists, it will be returned.
     *
     * @param name              Unique name of placement. The same name will be used in addapptr.com account.
     * @param supportsMainImage True if the native ads returned should have main image asset. Keep in mind that if main image is used, it has to be displayed.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    public static int createNativeAdPlacement(String name, boolean supportsMainImage) {
        logAATKitCall("CMD: createNativeAdPlacement(" + name + "," + supportsMainImage + ")");
        return adController.createNativeAdPlacement(name, supportsMainImage);
    }

    /**
     * Creates a new banner placement. If the banner placement of given name already exists, it will be returned.
     * <p>
     * <b>The placement will create a copy of the
     * configuration.</b> Any changes made to the configuration after placement is created will be ignored.
     *
     * @param name          Unique name of placement. The same name will be used in addapptr.com account.
     * @param configuration The configuration for this placement. The placement will ignore any changes made to configuration after it was created.
     * @return Banner placement instance, or null if the placement cannot be created.
     */
    public static BannerPlacement createBannerPlacement(String name, BannerConfiguration configuration) {
        logAATKitCall("CMD: createInFeedBannerPlacement(" + name + "," + configuration + ")");
        return adController.createBannerPlacement(name, configuration);
    }

    /**
     * Creates a new rewarded video placement. If the rewarded video ad placement of given name already exists, it will be returned.
     * <b>Only one Rewarded Video placement can be used within the app.</b>
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    public static int createRewardedVideoPlacement(String name) {
        logAATKitCall("CMD: createRewardedVideoPlacement(" + name + ")");
        return adController.createRewardedVideoPlacement(name);
    }

    /**
     * Enables automatic reloading of placement. Autoreloader will use reload time configured on addapptr.com account or fallback to default {@value #BANNER_DEFAULT_RELOAD_INTERVAL_IN_SECONDS} seconds.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     */
    public static void startPlacementAutoReload(int placementId) {
        logAATKitCall("CMD: startPlacementAutoReload(" + placementId + ")");
        adController.startPlacementAutoReload(placementId);
    }

    /**
     * Enables automatic reloading of placement and sets custom reload time. This reload time will be used instead of time configured on addapptr.com account.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param seconds     Interval between ad reloads. Should be a value from {@value #BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS} to {@value #BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS} seconds.
     */
    public static void startPlacementAutoReload(int placementId, int seconds) {
        logAATKitCall("CMD: startPlacementAutoReload(" + placementId + "," + seconds + ")");
        adController.startPlacementAutoReload(placementId, seconds);
    }

    /**
     * Disables automatic reloading of placement.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     */
    public static void stopPlacementAutoReload(int placementId) {
        logAATKitCall("CMD: stopPlacementAutoReload(" + placementId + ")");
        adController.stopPlacementAutoReload(placementId);
    }

    /**
     * Sets reload time that will be used instead of time configured on addapptr.com account.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param seconds     Interval between ad reloads. Should be a value from {@value #BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS} to {@value #BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS} seconds.
     */
    public static void setPlacementAutoreloadInterval(int placementId, int seconds) {
        logAATKitCall("CMD: setPlacementAutoreloadInterval(" + placementId + "," + seconds + ")");
        adController.setPlacementAutoreloadInterval(placementId, seconds);
    }

    /**
     * Requests placement reload. Works only if automatic reloading is disabled. For native ads, it might return false if current limit for native ads loading in parallel is reached.
     * <p>
     * For banners it will not cause new banner to show more often than set banner refresh interval (default: 30s) - instead, it will set a timer to reload and show next ad when the refresh interval passes. To force ad reload and display before this delay, use the method {@link #reloadPlacement(int, boolean)} instead.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @return True if call ends in causing a placement to reload, false otherwise.
     * @see #reloadPlacement(int, boolean)
     */
    public static boolean reloadPlacement(int placementId) {
        logAATKitCall("CMD: reloadPlacement(" + placementId + ")");
        return adController.reloadPlacement(placementId, false);
    }

    /**
     * Requests placement reload. Works only if automatic reloading is disabled. For native ads, it might return false if current limit for native ads loading in parallel is reached.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param forceLoad   True if you want to be able to reload and show new banner before set banner reload interval (default: 30s) between reloads, false otherwise.
     * @return True if call ends in causing a placement to reload, false otherwise.
     * @see #reloadPlacement(int)
     */
    public static boolean reloadPlacement(int placementId, boolean forceLoad) {
        logAATKitCall("CMD: reloadPlacement(" + placementId + "," + forceLoad + ")");
        return adController.reloadPlacement(placementId, forceLoad);
    }

    /**
     * Returns placement view. Works only for banner placements.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @return Placement view or null if {@code placementId} is not valid.
     */
    public static View getPlacementView(int placementId) {
        logAATKitCall("CMD: getPlacementView(" + placementId + ")");
        return adController.getPlacementView(placementId);
    }


    /**
     * Binds the native ad instance with given ViewGroup. Needed for click handling and tracking. Please note that this method will not work correctly with Facebook Audience Network ads.
     *
     * @param nativeAd Native ad instance.
     * @param layout   ViewGroup used to render the native ad.
     * @deprecated This method is deprecated and will be removed in the future. Please use {@link #attachNativeAdToLayout(NativeAdData, ViewGroup, View, View, View)} instead.
     */
    @Deprecated
    public static void attachNativeAdToLayout(NativeAdData nativeAd, ViewGroup layout) {
        logAATKitCall("CMD: attachNativeAdToLayout(" + nativeAd + ", " + layout.getClass().getSimpleName() + ")");
        if (Logger.isLoggable(Log.WARN)) {
            Logger.w(AATKit.class, "");
        }
        adController.attachNativeAdToLayout(nativeAd, layout, null, null, null);
    }

    /**
     * Binds the native ad instance with given ViewGroup. Needed for click handling and tracking.
     *
     * @param nativeAd      Native ad instance.
     * @param layout        ViewGroup used to render the native ad.
     * @param mainImageView View used to show the main image of the ad. Can be null.
     * @param iconView      View used to show the icon of the native ad.
     * @deprecated This method is deprecated and will be removed in the future. Please use {@link #attachNativeAdToLayout(NativeAdData, ViewGroup, View, View, View)} instead.
     */
    @Deprecated
    public static void attachNativeAdToLayout(NativeAdData nativeAd, ViewGroup layout, View mainImageView, View iconView) {
        String mainImageClass = mainImageView == null ? "null" : mainImageView.getClass().getSimpleName();
        String iconViewClass = iconView == null ? "null" : iconView.getClass().getSimpleName();
        logAATKitCall("CMD: attachNativeAdToLayout(" + nativeAd + ", " + layout.getClass().getSimpleName() + ", " + mainImageClass + ", " + iconViewClass + ")");
        adController.attachNativeAdToLayout(nativeAd, layout, mainImageView, iconView, null);
    }

    /**
     * Binds the native ad instance with given ViewGroup. Needed for click handling and tracking.
     *
     * @param nativeAd      Native ad instance.
     * @param layout        ViewGroup used to render the native ad.
     * @param mainImageView View used to show the main image of the ad. Can be null.
     * @param iconView      View used to show the icon of the native ad.
     * @param CTAView       View used to show the Call To Action of the native ad. Can be null.
     */
    public static void attachNativeAdToLayout(NativeAdData nativeAd, ViewGroup layout, View mainImageView, View iconView, View CTAView) {
        String mainImageClass = mainImageView == null ? "null" : mainImageView.getClass().getSimpleName();
        String iconViewClass = iconView == null ? "null" : iconView.getClass().getSimpleName();
        String ctaViewClass = CTAView == null ? "null" : CTAView.getClass().getSimpleName();
        logAATKitCall("CMD: attachNativeAdToLayout(" + nativeAd + ", " + layout.getClass().getSimpleName() + ", " + mainImageClass + ", " + iconViewClass + ", " + ctaViewClass + ")");
        adController.attachNativeAdToLayout(nativeAd, layout, mainImageView, iconView, CTAView);
    }

    /**
     * Removes the binding between native ad and ViewGroup. Should be called when the native ad will no longer be presented and should be destroyed.
     *
     * @param nativeAd Native ad instance.
     */
    public static void detachNativeAdFromLayout(NativeAdData nativeAd) {
        logAATKitCall("CMD: detachNativeAdFromLayout(" + nativeAd + ")");
        adController.detachNativeAdFromLayout(nativeAd);
    }

    /**
     * Returns the title of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with title asset of the ad, or null if it is not available.
     */
    public static String getNativeAdTitle(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdTitle(" + nativeAd + ")");
        return adController.getNativeAdTitle(nativeAd);
    }

    /**
     * Returns the description of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with description asset of the ad, or null if it is not available.
     */
    public static String getNativeAdDescription(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdDescription(" + nativeAd + ")");
        return adController.getNativeAdDescription(nativeAd);
    }

    /**
     * Returns the call to action of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with call to action asset of the ad, or null if it is not available.
     */
    public static String getNativeAdCallToAction(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdCallToAction(" + nativeAd + ")");
        return adController.getNativeAdCallToAction(nativeAd);
    }

    /**
     * Returns the URL of the image asset of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with URL of the image asset of the ad, or null if it is not available.
     */
    public static String getNativeAdImageUrl(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdImageUrl(" + nativeAd + ")");
        return adController.getNativeAdImageUrl(nativeAd);
    }

    /**
     * Returns the URL of the icon asset of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with URL of the icon asset of the ad, or null if it is not available.
     */
    public static String getNativeAdIconUrl(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdIconUrl(" + nativeAd + ")");
        return adController.getNativeAdIconUrl(nativeAd);
    }

    /**
     * Returns the rating asset of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return NativeAdRating instance containing both value and scale of rating, or null if it is not available.
     */
    public static NativeAd.NativeAdRating getNativeAdRating(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdRating(" + nativeAd + ")");
        return adController.getNativeAdRating(nativeAd);
    }

    /**
     * Returns the view with branding logo or ad information related to the ad network providing the native ad. Some networks like Facebook Audience Network require this special view to be visible on native ads.
     *
     * @param nativeAd Native ad insance.
     * @return View that should be added to native ad layout, or null if it is not available.
     */
    public static View getNativeAdBrandingLogo(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdBrandingLogo(" + nativeAd + ")");
        return adController.getNativeAdBrandingLogo(nativeAd);
    }

    /**
     * Returns the advertiser asset of native ad (not the same as ad network providing it).
     *
     * @param nativeAd Native ad instance.
     * @return String with advertiser asset of the ad, or null if it is not available.
     */
    public static String getNativeAdAdvertiser(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdAdvertiser(" + nativeAd + ")");
        return adController.getNativeAdAdvertiser(nativeAd);
    }

    /**
     * Returns the type of native ad. Only some networks support this, default value is {@link com.intentsoftware.addapptr.ad.NativeAd.Type#UNKNOWN}
     *
     * @param nativeAd Native ad instance.
     * @return Enum value representing the type of given native ad.
     * @deprecated For Google Native Ads, it is now recommended to use the Unified ad type. See our documentation on native ads integration to learn more.
     */
    @Deprecated
    public static NativeAd.Type getNativeAdType(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdType(" + nativeAd + ")");
        return adController.getNativeAdType(nativeAd);
    }

    /**
     * Returns the ad network providing given native ad
     *
     * @param nativeAd Native ad instance.
     * @return Enum value representing the ad network providing the ad.
     */
    public static AdNetwork getNativeAdNetwork(NativeAdData nativeAd) {
        logAATKitCall("CMD: getNativeAdNetwork(" + nativeAd + ")");
        return adController.getNativeAdNetwork(nativeAd);
    }

    /**
     * Returns if the native ad has expired and shall no longer be used.
     *
     * @param nativeAd Native ad instance.
     * @return True if native ad has expired, false otherwise.
     */
    public static boolean isNativeAdExpired(NativeAdData nativeAd) {
        logAATKitCall("CMD: isNativeAdExpired(" + nativeAd + ")");
        return adController.isNativeAdExpired(nativeAd);
    }

    /**
     * Returns if the native ad is ready to be displayed.
     *
     * @param nativeAd Native ad instance.
     * @return True if native ad is ready, false otherwise.
     */
    public static boolean isNativeAdReady(NativeAdData nativeAd) {
        logAATKitCall("CMD: isNativeAdReady(" + nativeAd + ")");
        return adController.isNativeAdReady(nativeAd);
    }

    /**
     * Indicates that the application wants to display a native ad for given native ad placement. Used in reporting.
     *
     * @param placementId Placement identifier obtained from
     *                    {@link #createPlacement(String, PlacementSize)}.
     * @deprecated Please use {@link #reportAdSpaceForPlacement(int)} instead.
     */
    @Deprecated
    public static void reportAdSpaceForNativePlacement(int placementId) {
        logAATKitCall("CMD: reportAdSpaceForNativePlacement(" + placementId + ")");
        adController.reportAdSpaceForPlacement(placementId);
    }

    /**
     * Indicates that the application wants to display a native or VAST ad for given placement. Used in reporting.
     *
     * @param placementId Placement identifier obtained from
     *                    {@link #createPlacement(String, PlacementSize)}.
     */
    public static void reportAdSpaceForPlacement(int placementId) {
        logAATKitCall("CMD: reportAdSpaceForPlacement(" + placementId + ")");
        adController.reportAdSpaceForPlacement(placementId);
    }

    /**
     * Returns how many ads are currently loading for given native ad placement.
     *
     * @param placementId Placement identifier obtained from
     *                    {@link #createPlacement(String, PlacementSize)}.
     * @return Number of ads that are currently loading for given placement.
     */
    public static int currentlyLoadingNativeAdsOnPlacement(int placementId) {
        logAATKitCall("CMD: currentlyLoadingNativeAdsOnPlacement(" + placementId + ")");
        return adController.currentlyLoadingNativeAdsOnPlacement(placementId);
    }

    /**
     * Checks if desired impression cap (set on AddApptr Dashboard) is reached for given native ad placement.
     *
     * @param placementId Placement identifier obtained from
     *                    {@link #createPlacement(String, PlacementSize)}.
     * @return True if impression cap is reached for this placement, false otherwise.
     * @deprecated This method is deprecated and will be removed in the future. Please use {@link #isFrequencyCapReachedForPlacement(int)} method instead.
     */
    @Deprecated
    public static boolean isFrequencyCapReachedForNativePlacement(int placementId) {
        logAATKitCall("CMD: isFrequencyCapReachedForNativePlacement(" + placementId + ")");
        return adController.isFrequencyCapReachedForPlacement(placementId);
    }

    /**
     * Checks if desired impression cap (set on AddApptr Dashboard) is reached for given native ad, fullscreen, rewarded video or VAST placement.
     * Note: this method will ALWAYS return "false" for other placement types.
     *
     * @param placementId Placement identifier obtained from
     *                    {@link #createPlacement(String, PlacementSize)}.
     * @return True if impression cap is reached for given placement, false otherwise.
     */
    public static boolean isFrequencyCapReachedForPlacement(int placementId) {
        logAATKitCall("CMD: isFrequencyCapReachedForPlacement(" + placementId + ")");
        return adController.isFrequencyCapReachedForPlacement(placementId);
    }

    /**
     * Returns the instance of native ad for given native ad placement.
     *
     * @param placementId Placement identifier obtained from
     *                    {@link #createPlacement(String, PlacementSize)}.
     * @return Native ad instance if it is loaded for given placement, null otherwise.
     */
    public static NativeAdData getNativeAd(int placementId) {
        logAATKitCall("CMD: getNativeAd(" + placementId + ")");
        return adController.getNativeAd(placementId);
    }

    /**
     * Returns identifier of placement with given name, or -1 if placement with given name does not exist.
     *
     * @param placementName Placent name, used when creating placement with {@link #createPlacement(String, PlacementSize)} method.
     * @return Placement identifier, or -1 if placement does not exist.
     * @deprecated This method is deprecated and will be removed in the future. Please use {@link #getPlacementIdForName(String)} method instead.
     */
    @Deprecated
    public static int getPlacmentIdForName(String placementName) {
        logAATKitCall("CMD: getPlacmentIdForName(" + placementName + ")");
        return adController.findPlacementIdByRealName(placementName);
    }

    /**
     * Returns identifier of placement with given name, or -1 if placement with given name does not exist.
     *
     * @param placementName Placent name, used when creating placement with {@link #createPlacement(String, PlacementSize)} method.
     * @return Placement identifier, or -1 if placement does not exist.
     */
    public static int getPlacementIdForName(String placementName) {
        logAATKitCall("CMD: getPlacementIdForName(" + placementName + ")");
        return adController.findPlacementIdByRealName(placementName);
    }

    /**
     * Returns true if there is an ad loaded for given placementId.
     *
     * @param placementId Placement identifier obtained from
     *                    {@link #createPlacement(String, PlacementSize)}.
     * @return True if there is an ad loaded for given placementId.
     */
    public static boolean hasAdForPlacement(int placementId) {
        logAATKitCall("CMD: hasAdForPlacement(" + placementId + ")");
        return adController.hasAdForPlacement(placementId);
    }

    /**
     * Shows interstitial ad if ad is ready.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @return True if showing interstitial was successful, false otherwise.
     */
    public static boolean showPlacement(int placementId) {
        logAATKitCall("CMD: showPlacement(" + placementId + ")");
        return adController.showPlacement(placementId);
    }

    /**
     * Allows to enable or disable selected ad networks. By default all networks are enabled.
     *
     * @param network Ad network.
     * @param enabled True to enable, false to disable.
     */
    public static void setNetworkEnabled(AdNetwork network, boolean enabled) {
        logAATKitCall("CMD: setNetworkEnabled(" + network + "," + enabled + ")");
        SupportedNetworks.setNetworkEnabled(network, enabled);
    }

    /**
     * Returns true if ad network is enabled, false otherwise.
     *
     * @param network Ad network.
     * @return True if ad network is enabled, false otherwise.
     */
    public static boolean isNetworkEnabled(AdNetwork network) {
        logAATKitCall("CMD: isNetworkEnabled(" + network + ")");
        return SupportedNetworks.isNetworkEnabled(network);
    }

    /**
     * Sets placement default image. This image will be shown in placement when no ad is available.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param image       The bitmap to set.
     */
    public static void setPlacementDefaultImageBitmap(int placementId, Bitmap image) {
        logAATKitCall("CMD: setPlacementDefaultImageBitmap(" + placementId + "," + image + ")");
        adController.setPlacementDefaultImage(placementId, image);
    }

    /**
     * Sets placement default image. This image will be shown in placement when no ad is available.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param resId       The identifier of the resource.
     */
    public static void setPlacementDefaultImageResource(int placementId, int resId) {
        logAATKitCall("CMD: setPlacementDefaultImageResource(" + placementId + "," + resId + ")");
        adController.setPlacementDefaultImageResource(placementId, resId);
    }

    /**
     * Get option from AATKit. Options can be obtained from the server or set using the {@link #setOption(String, String)} method.
     *
     * @param optionName The name of the option to be checked.
     * @return Value of the option or null if it is not set.
     */
    public static String getOption(String optionName) {
        logAATKitCall("CMD: getOption(" + optionName + ")");
        return AdController.getOption(optionName);
    }

    /**
     * Convenience method for checking if option is enabled in AATKit. Options can be obtained from the server or set using the {@link #setOption(String, String)} method.
     *
     * @param optionName The name of the option to be checked.
     * @return True if option value is "Yes", false otherwise.
     */
    public static boolean isOptionEnabled(String optionName) {
        logAATKitCall("CMD: isOptionEnabled(" + optionName + ")");
        return AdController.isOptionEnabled(optionName);
    }

    /**
     * Set option in AATKit. Options can also be obtained from the server.
     *
     * @param optionName  The name of the option to be set.
     * @param optionValue The value of the option to be set.
     */
    public static void setOption(String optionName, String optionValue) {
        logAATKitCall("CMD: setOption(" + optionName + "," + optionName + ")");
        adController.setOption(optionName, optionValue);
    }

    /**
     * Sets the targeting information for the application. This information will be used only if no placement-specific targeting is available.
     *
     * @param info Map with targeting information
     * @see #setTargetingInfo(int, Map)
     */
    public static void setTargetingInfo(Map<String, List<String>> info) {
        logAATKitCall("CMD: setTargetingInfo(" + info + ")");
        adController.setTargetingInfo(null, info);
    }

    /**
     * Sets the targeting information for the given placement. Information provided for placement overrides targeting information for application set by the {@link #setTargetingInfo(Map)}.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param info        Map with targeting information
     */
    public static void setTargetingInfo(int placementId, Map<String, List<String>> info) {
        logAATKitCall("CMD: setTargetingInfo(" + placementId + ", " + info + ")");
        adController.setTargetingInfo(placementId, info);
    }

    /**
     * Sets the content targeting url for the application. This information will be used only if no placement-specific targeting is available.
     *
     * @param targetingUrl The targeting url
     * @see #setContentTargetingUrl(int, String)
     */
    public static void setContentTargetingUrl(String targetingUrl) {
        logAATKitCall("CMD: setContentTargetingUrl(" + targetingUrl + ")");
        adController.setContentTargetingUrl(null, targetingUrl);
    }

    /**
     * Sets the content targeting url for the given placement. Information provided for placement overrides targeting information for application set by the {@link #setContentTargetingUrl(String)}.
     *
     * @param placementId  Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param targetingUrl The targeting url
     */
    public static void setContentTargetingUrl(int placementId, String targetingUrl) {
        logAATKitCall("CMD: setContentTargetingUrl(" + placementId + ", " + targetingUrl + ")");
        adController.setContentTargetingUrl(placementId, targetingUrl);
    }

    /**
     * Adds an ad network to the list of ad networks that receive targeting keywords (if any set).
     * If no ad networks are added, any set keywords will be delivered to all ad networks supporting keyword targeting.
     *
     * @param network Chosen ad network.
     */
    public static void addAdNetworkForKeywordTargeting(AdNetwork network) {
        logAATKitCall("CMD: addAdNetworkForKeywordTargeting(" + network + ")");
        adController.addAdNetworkForKeywordTargeting(network);
    }

    /**
     * Removes an ad network from the list of ad networks that receive targeting keywords (if any set).
     * If no ad networks are added to the list, any set keywords will be delivered to all ad networks supporting keyword targeting.
     *
     * @param network Chosen ad network.
     */
    public static void removeAdNetworkForKeywordTargeting(AdNetwork network) {
        logAATKitCall("CMD: removeAdNetworkForKeywordTargeting(" + network + ")");
        adController.removeAdNetworkForKeywordTargeting(network);
    }

    /**
     * Sets gravity for ads that don't fill entire placement area. Works only for banner placements.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param gravity     The {@link Gravity} to set.
     */
    public static void setPlacementContentGravity(int placementId, int gravity) {
        logAATKitCall("CMD: setPlacementContentGravity(" + placementId + "," + gravity + ")");
        adController.setPlacementContentGravity(placementId, gravity);
    }

    /**
     * Allows setting of ad rules that will be used before real rules from the server are downloaded.
     *
     * @param rules String containing the rules to be used.
     */
    public static void setInitialRules(String rules) {
        logAATKitCall("CMD: setInitialRules(" + rules + ")");
        adController.setInitialRules(rules);
    }

    /**
     * Allows the AATKit to preserve last downloaded ad rules when the application is closed. Such rules will be re-used next time the application is started, before new ones get downloaded.
     *
     * @param enabled True to enable, false to disable.
     */
    public static void setRuleCachingEnabled(boolean enabled) {
        logAATKitCall("CMD: setRuleCachingEnabled(" + enabled + ")");
        adController.setRuleCachingEnabled(enabled);
    }

    /**
     * Allows to set log level from code.
     *
     * @param logLevel Desired log level, as in {@link android.util.Log} class.
     */
    public static void setLogLevel(int logLevel) {
        ServerLogger.log("CMD: setLogLevel(" + logLevel + ")");
        adController.setLogLevel(logLevel);
    }

    /**
     * Reports the VAST ad impression.
     *
     * @param data The data for a VAST ad which has been impressed.
     */
    public static void reportVASTImpression(VASTAdData data) {
        logAATKitCall("CMD: reportVASTImpression(" + data + ")");
        adController.reportVASTImpression(data);
    }

    /**
     * Reports the VAST ad viewable impression.
     *
     * @param data The VAST ad data for which the viewable impression should be counted.
     */
    public static void reportVASTViewableImpression(VASTAdData data) {
        logAATKitCall("CMD: reportVASTViewableImpression(" + data + ")");
        adController.reportVASTViewableImpression(data);
    }

    /**
     * Reports the VAST ad click.
     *
     * @param data The data for a VAST ad which has been clicked.
     */
    public static void reportVASTClick(VASTAdData data) {
        logAATKitCall("CMD: reportVASTClick(" + data + ")");
        adController.reportVASTClick(data);
    }

    /**
     * Sets VAST ad request parameters.
     *
     * @param placementId Placement identifier obtained from {@link #createPlacement(String, PlacementSize)}.
     * @param parameters  Parameters to set for a given placement identifier.
     */
    public static void setVASTRequestParameters(int placementId, VASTRequestParameters parameters) {
        logAATKitCall("CMD: setVASTRequestParameters(" + placementId + ", " + parameters + ")");
        adController.setVASTRequestParameters(placementId, parameters);
    }

    //region non-public methods

    private static void logAATKitCall(String call) {
        if (adController != null && adController.shouldLogAATKitCalls()) {
            ServerLogger.log(call);
        }
        if (Logger.isLoggable(Log.VERBOSE)) {
            String command = call.replaceFirst("^CMD:\\s*", "");
            Logger.v(AATKit.class, command);
        }
    }

    static Pair<BannerPlacement, Boolean> createBannerPlacementForCache(String placementName, BannerConfiguration configuration, BannerCache cache) {
        return adController.createBannerPlacementForCache(placementName, configuration, cache);
    }

    static void destroyBannerCache(BannerCache cache) {
        adController.destroyBannerCache(cache);
    }

    // methods below are intended only for use by the plugins.

    static void destroy() {
        adController.destroy();
        adController = null;
    }

    static boolean isInitialized() {
        return adController != null;
    }

    /**
     * Shows AATKit debug screen.
     */
    static void showDebugDialog() {
        logAATKitCall("CMD: showDebugDialog()");
        adController.showDebugDialog();
    }

    //endregion

}
