package com.intentsoftware.addapptr;

import android.app.Application;
import android.util.Log;

import com.intentsoftware.addapptr.module.Logger;

import androidx.annotation.NonNull;

/**
 * Used to set up AATKit.
 */
@SuppressWarnings({"WeakerAccess"})
public class AATKitConfiguration extends AATKitRuntimeConfiguration {
    private String initialRules;
    private boolean shouldCacheRules = true;
    private boolean shouldSkipRules;
    private String alternativeBundleId;
    private boolean shouldReportUsingAlternativeBundleId = true;
    private int testModeAccountId;
    private AATKit.Delegate delegate;
    private boolean useDebugShake = true;
    private Application application;

    /**
     * Creates the default Configuration object to be used when initializing AATKit.
     *
     * @param application Reference to {@link Application}.
     */
    public AATKitConfiguration(Application application) {
        if (application == null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(AATKit.class, "Application cannot be null!");
            }
            return;
        }

        this.application = application;
    }

    @NonNull
    @Override
    public String toString() {
        return "AATKitConfiguration{" +
                "initialRules='" + initialRules + '\'' +
                ", shouldCacheRules=" + shouldCacheRules +
                ", alternativeBundleId='" + alternativeBundleId + '\'' +
                ", shouldReportUsingAlternativeBundleId=" + shouldReportUsingAlternativeBundleId +
                ", testModeAccountId=" + testModeAccountId +
                ", delegate=" + delegate +
                ", useDebugShake=" + useDebugShake +
                ", application=" + application +
                "} " + super.toString();
    }

    String getInitialRules() {
        return initialRules;
    }

    /**
     * Allows setting of ad rules that will be used before real rules from the server are downloaded.
     *
     * @param initialRules String containing the rules to be used.
     */
    public void setInitialRules(String initialRules) {
        this.initialRules = initialRules;
    }

    boolean isShouldCacheRules() {
        return shouldCacheRules;
    }

    /**
     * Allows the AATKit to preserve last downloaded ad rules when the application is closed.
     * Such rules will be re-used next time the application is started, before new ones get downloaded.
     * Enabled by default.
     *
     * @param shouldCacheRules True to enable, false to disable.
     */
    public void setShouldCacheRules(boolean shouldCacheRules) {
        this.shouldCacheRules = shouldCacheRules;
    }

    String getAlternativeBundleId() {
        return alternativeBundleId;
    }

    /**
     * Sets the fake bundle ID for testing purposes. It should only be used during development.
     * Cannot be used together with classic test mode ({@link #setTestModeAccountId(int)}).
     *
     * @param alternativeBundleId Bundle ID to be used during testing.
     */
    public void setAlternativeBundleId(String alternativeBundleId) {
        if (this.alternativeBundleId != null) {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(this, "Alternative bundle ID was already set! It will be overriden.");
            }
        }
        if (testModeAccountId != 0) {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(this, "Test mode was already enabled! It will be overriden by this bundle ID.");
            }
        }

        this.alternativeBundleId = alternativeBundleId;
    }

    boolean isShouldReportUsingAlternativeBundleId() {
        return shouldReportUsingAlternativeBundleId;
    }

    /**
     * If used together with {@link #setAlternativeBundleId(String)} allows to set if the same fake bundle ID should be used in reporting.
     * True by default. If set to false, real bundle ID will be used in reporting even if fake one is used for testing.
     *
     * @param shouldReportUsingAlternativeBundleId If the fake bundleId should be used in reporting.
     */
    public void setShouldReportUsingAlternativeBundleId(boolean shouldReportUsingAlternativeBundleId) {
        this.shouldReportUsingAlternativeBundleId = shouldReportUsingAlternativeBundleId;
    }

    int getTestModeAccountId() {
        return testModeAccountId;
    }

    /**
     * Enables AATKit test ads that should be for testing - only during development.
     * Cannot be used with alternative bundle ID ({@link #setAlternativeBundleId(String)}).
     *
     * @param testModeAccountId Test application id obtained from addapptr.com account.
     */
    public void setTestModeAccountId(int testModeAccountId) {
        if (this.alternativeBundleId != null) {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(this, "Alternative bundle ID was already set! The test mode account ID will be ignored.");
            }
        }
        if (this.testModeAccountId != 0) {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(this, "Test mode was already enabled! Old value for test account ID will be overriden.");
            }
        }

        this.testModeAccountId = testModeAccountId;
    }

    AATKit.Delegate getDelegate() {
        return delegate;
    }

    /**
     * Sets the delegate notifying about AATKit events.
     *
     * @param delegate Delegate allowing to listen for AATKit events.
     */
    public void setDelegate(AATKit.Delegate delegate) {
        this.delegate = delegate;
    }

    boolean isUseDebugShake() {
        return useDebugShake;
    }

    /**
     * Sets if the debug screen should be displayed after shaking the device. Enabled by default.
     *
     * @param useDebugShake True to enable, false to disable.
     */
    public void setUseDebugShake(boolean useDebugShake) {
        this.useDebugShake = useDebugShake;
    }

    /**
     * Sets if networks without TCF2 consent should be skipped (only works if TCF2 compliant CMP is used). False by default.
     *
     * @param shouldSkipRules True to enable rule skipping, false to disable.
     */
    public void setShouldSkipRules(boolean shouldSkipRules) {
        this.shouldSkipRules = shouldSkipRules;
    }

    Application getApplication() {
        return application;
    }

    boolean isShouldSkipRules() {
        return shouldSkipRules;
    }
}
