package com.intentsoftware.addapptr;

import android.app.Application;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.util.Pair;

import com.intentsoftware.addapptr.module.Logger;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

/**
 * A cache of automatically preloaded banner ads. The cache will always try to have a defined amount of banners available for immediate handout to the app whenever they are needed.
 * <b>The BannerCache needs to be destroyed when no longer needed.</b>
 */
public class BannerCache {

    private static final String BANNER_CACHE_RELOAD_INTERVAL_METADATA_TAG = "com.intentsoftware.addapptr.banner_cache_reload_interval";
    private static final int ON_FAILED_BANNER_RELOAD_INTERVAL_DEFAULT = 10000;
    private static int onFailedBannerReloadInterval = ON_FAILED_BANNER_RELOAD_INTERVAL_DEFAULT;

    private final BannerCacheConfiguration configuration;
    private final BannerPlacement bannerPlacement;
    private final Handler handler = new Handler();

    private final Queue<BannerPlacementLayout> loadedBanners = new LinkedList<>();
    private final List<BannerRequest> runningRequests = new ArrayList<>();
    private final List<Runnable> bannerReloadRunnables = new ArrayList<>(); //list for runnables for retrying failed banner requests

    private long consumptionTimestamp;

    private boolean destroyed;
    private boolean shouldNotifyDelegate = true;


    //called by AdController, as we need an Context instance
    static void init(Application application) {
        try {
            ApplicationInfo ai = application.getPackageManager().getApplicationInfo(application.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = ai.metaData;
            int reloadInterval = bundle.getInt(BANNER_CACHE_RELOAD_INTERVAL_METADATA_TAG);
            if (reloadInterval > 0) {
                if (Logger.isLoggable(Log.VERBOSE)) {
                    Logger.v(BannerCache.class, "Found value:" + reloadInterval + " for BannerCache banner reload interval.");
                }
                onFailedBannerReloadInterval = reloadInterval * 1000;
            } else {
                if (Logger.isLoggable(Log.VERBOSE)) {
                    Logger.v(BannerCache.class, "No value for BannerCache banner reload interval found.");
                }
            }
        } catch (Exception e) {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(BannerCache.class, "Exception when looking for BannerCache banner reload interval in manifest", e);
            }
        }
    }

    //region Public API

    /**
     * Creates a cache of automatically preloaded banner ads.
     *
     * @param cacheConfiguration {@link BannerCacheConfiguration} instance used to configure the cache.
     */
    public BannerCache(BannerCacheConfiguration cacheConfiguration) {
        this.configuration = new BannerCacheConfiguration(cacheConfiguration);

        BannerConfiguration configuration = new BannerConfiguration();
        configuration.setManualAdSpaceCounting(true);
        Pair<BannerPlacement, Boolean> placementBooleanPair = AATKit.createBannerPlacementForCache(cacheConfiguration.getPlacementName(), configuration, this);

        bannerPlacement = placementBooleanPair.first;

        if (bannerPlacement == null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "Failed to create banner placement with name:" + cacheConfiguration.getPlacementName() + " for banner cache:" + toString() + ", cache will not work.");
            }
        } else if (Logger.isLoggable(Log.VERBOSE)) {
            Logger.v(this, "Created banner cache:" + toString() + " with placement name:" + cacheConfiguration.getPlacementName());
        }

        if (placementBooleanPair.second) {
            onResume();
        }
    }

    /**
     * Creates a cache of automatically preloaded banner ads.
     *
     * @param placementName                  The name of the banner placement that will be created. <b>The placement will be created by the cache and should not be created manually.</b>
     * @param delegate                       Optional cache delegate. Can be null.
     * @param size                           Defines how many preloaded banners should be available in the cache. Should be smaller than 5.
     * @param shouldCacheAdditionalAdAtStart Defines if the cache should load additional ad at the beginning.
     * @param requestConfiguration           The configuration that will be used when requesting new banners. Can not be null.
     * @deprecated This method is deprecated and will be removed in the future. Use {@link #BannerCache(BannerCacheConfiguration)} instead.
     */
    @Deprecated
    public BannerCache(String placementName, CacheDelegate delegate, int size, boolean shouldCacheAdditionalAdAtStart, BannerRequest requestConfiguration) {
        this(prepareConfiguration(placementName, size, delegate, shouldCacheAdditionalAdAtStart, requestConfiguration)); //"this" call must be the first one in constructor body, thus - use helper static method
    }

    private static BannerCacheConfiguration prepareConfiguration(String placementName, int size, CacheDelegate delegate, boolean shouldCacheAdditionalAdAtStart, BannerRequest requestConfiguration) {
        BannerCacheConfiguration configuration = new BannerCacheConfiguration(placementName, size);
        configuration.setDelegate(delegate);
        configuration.setShouldCacheAdditionalAdAtStart(shouldCacheAdditionalAdAtStart);
        configuration.setRequestConfiguration(requestConfiguration);
        configuration.setMinimumDelay(0);
        return configuration;
    }

    /**
     * Updates the configuration that will be used when requesting new banners.
     *
     * @param requestConfiguration {@link BannerRequest} instance. Can not be null.
     * @param shouldRefresh        True if the whole cache should be re-loaded with new banner request configuration, false if new configuration should only be used for new requests.
     */
    @SuppressWarnings("unused")
    public void updateRequestConfiguration(BannerRequest requestConfiguration, boolean shouldRefresh) {
        if (checkDestroyedOrFailedToInitialize("updateRequestConfiguration")) return;
        if (Logger.isLoggable(Log.VERBOSE)) {
            Logger.v(this, "Updating request configuration with configuration:" + requestConfiguration + ", shouldRefresh:" + shouldRefresh + " for cache:" + toString());
        }

        this.configuration.setRequestConfiguration(requestConfiguration);
        if (shouldRefresh) {
            shouldNotifyDelegate = true; //we want to inform the delegate again
            clearCache();
            cancelRunningRequests();
            fillCache();
        }
    }

    /**
     * Sets the optional cache delegate.
     *
     * @param delegate {@link CacheDelegate} instance.
     * @deprecated This method is deprecated and will be removed in the future. Use {@link BannerCacheConfiguration#setDelegate(CacheDelegate)} instead.
     */
    @SuppressWarnings("unused")
    @Deprecated
    public void setCacheDelegate(CacheDelegate delegate) {
        if (checkDestroyedOrFailedToInitialize("setCacheDelegate")) return;
        if (Logger.isLoggable(Log.VERBOSE)) {
            Logger.v(this, "Setting delegate:" + delegate + " for cache:" + toString());
        }

        configuration.setDelegate(delegate);
    }

    /**
     * Returns an instance of {@link BannerPlacementLayout} to be used within the app. Also automatically counts an ad space.
     * <p>
     * <b>BannerCache will no longer hold any references to returned banners, and they need to be destroyed manually by the app.</b>
     * <p>
     * This method respects the frequency capping, set by {@link BannerCacheConfiguration#setMinimumDelay(int)}
     *
     * @return {@link BannerPlacementLayout} instance.
     */
    @SuppressWarnings("unused")
    public synchronized BannerPlacementLayout consume() {
        return consume(false);
    }

    /**
     * Returns an instance of {@link BannerPlacementLayout} to be used within the app. Also automatically counts an ad space.
     * <b>BannerCache will no longer hold any references to returned banners, and they need to be destroyed manually by the app.</b>
     *
     * @param force true if cache should try to return banner ignoring the frequency capping set by {@link BannerCacheConfiguration#setMinimumDelay(int)}
     * @return {@link BannerPlacementLayout} instance.
     */
    public synchronized BannerPlacementLayout consume(boolean force) {
        if (checkDestroyedOrFailedToInitialize("consume")) return null;
        if (Logger.isLoggable(Log.VERBOSE)) {
        	Logger.v(this, "Called consume method with param force=" + force);
        }

        long currentTime = System.currentTimeMillis();
        if ((currentTime - this.consumptionTimestamp >= configuration.getMinDelayMillis()) || force) {
            bannerPlacement.countAdSpace();
            BannerPlacementLayout banner = loadedBanners.poll();
            if (Logger.isLoggable(Log.VERBOSE)) {
                Logger.v(this, "Consuming banner from cache:" + toString() + ", returning:" + banner);
            }
            fillCache();
            if (banner != null) {
                this.consumptionTimestamp = currentTime;
            }
            return banner;
        } else {
            if (Logger.isLoggable(Log.VERBOSE)) {
            	Logger.v(this, "Minimum delay between \"consume\" calls not reached, returning null");
            }
            return null;
        }
    }

    /**
     * Destroys the BannerCache, clearing all preloaded banner ads and canceling pending reload requests. For proper memory management, it needs to be called when the BannerCache is no longer needed.
     * <b>Destroyed BannerCache can no longer be used.</b>
     */
    @SuppressWarnings("unused")
    public void destroy() {
        if (checkDestroyedOrFailedToInitialize("destroy")) return;
        if (Logger.isLoggable(Log.VERBOSE)) {
            Logger.v(this, "Destroying cache:" + toString());
        }

        AATKit.destroyBannerCache(this);
        cancelRunningRequests();
        clearCache();
        configuration.setDelegate(null);
        configuration.setRequestConfiguration(null);
        destroyed = true;
    }

    //endregion

    //region lifecycle handling

    void onResume() {
        fillCache();
    }

    void onPause() {
        cancelRunningRequests();
    }

    //endregion

    //region private methods

    private boolean checkDestroyedOrFailedToInitialize(String methodName) {
        if (bannerPlacement == null) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "Cannot execute method " + methodName + ", BannerCache was not created properly.");
            }
            return true;
        } else if (destroyed) {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(this, "Cannot execute method " + methodName + ", BannerCache is already destroyed!");
            }
        }
        return destroyed;
    }

    private synchronized void fillCache() {
        if (!((Placement) bannerPlacement).isActivityResumed()) {
            if (Logger.isLoggable(Log.DEBUG)) {
                Logger.d(this, "Cannot fill cache, activity is paused");
            }
            return;
        }

        int adsToLoad = configuration.getSize() - loadedBanners.size() - runningRequests.size() - bannerReloadRunnables.size();

        if (adsToLoad > 0) {
            if (configuration.shouldCacheAdditionalAdAtStart()) {
                adsToLoad++;
                configuration.setShouldCacheAdditionalAdAtStart(false);
            }
            if (Logger.isLoggable(Log.VERBOSE)) {
                Logger.v(this, "Banner cache " + toString() + " will request:" + adsToLoad + " banners.");
            }
            for (int i = 0; i < adsToLoad; i++) {
                requestAd();
            }
        }
    }

    private synchronized void clearCache() {
        for (BannerPlacementLayout banner : loadedBanners) {
            banner.destroy();
        }
        loadedBanners.clear();
    }

    private synchronized void cancelRunningRequests() {
        if (Logger.isLoggable(Log.VERBOSE)) {
        	Logger.v(this, "Canceling running requests");
        }

        List<BannerRequest> requestsToBeCanceled = new ArrayList<>(runningRequests); //needs a copy, as request.cancel ends up in modifying the list and thus ConcurrentModificationException
        runningRequests.clear();
        for (BannerRequest request : requestsToBeCanceled) {
            bannerPlacement.cancel(request);
        }

        for (Runnable runnable : bannerReloadRunnables) {
            handler.removeCallbacks(runnable);
        }
        bannerReloadRunnables.clear();
    }

    private synchronized void requestAd() {
        BannerRequest newRequest = new BannerRequest(configuration.getRequestConfiguration().getDelegate());
        newRequest.setBannerSizes(configuration.getRequestConfiguration().getBannerSizes());
        newRequest.setTargetingInformation(configuration.getRequestConfiguration().getTargetingInformation());
        newRequest.setContentTargetingUrl(configuration.getRequestConfiguration().getContentTargetingUrl());

        runningRequests.add(newRequest);
        bannerPlacement.requestAd(newRequest, createRequestCompletionListener(newRequest));
    }

    private BannerRequestCompletionListener createRequestCompletionListener(final BannerRequest request) {
        return new BannerRequestCompletionListener() {
            @Override
            public void onRequestCompleted(BannerPlacementLayout layout, BannerRequestError error) {
                synchronized (BannerCache.this) {
                    runningRequests.remove(request);
                    if (layout != null) {
                        if (Logger.isLoggable(Log.VERBOSE)) {
                            Logger.v(BannerCache.this, "Banner:" + layout + " added to cache:" + BannerCache.this.toString());
                        }
                        loadedBanners.add(layout);
                        if (shouldNotifyDelegate && configuration.getDelegate() != null) {
                            if (Logger.isLoggable(Log.VERBOSE)) {
                                Logger.v(BannerCache.this, "Banner cache:" + BannerCache.this.toString() + " notifying delegate:" + configuration.getDelegate());
                            }
                            configuration.getDelegate().firstBannerLoaded();
                        }
                        shouldNotifyDelegate = false;
                    } else {
                        if (Logger.isLoggable(Log.VERBOSE)) {
                            Logger.v(BannerCache.this, "Failed to load banner for queue, error:" + error.getMessage());
                        }
                        Runnable runnable = new Runnable() {
                            @Override
                            public void run() {
                                synchronized (BannerCache.this) {
                                    bannerReloadRunnables.remove(this);
                                    requestAd();
                                }
                            }
                        };
                        bannerReloadRunnables.add(runnable);
                        handler.postDelayed(runnable, onFailedBannerReloadInterval); //retry banner reload after given time
                    }
                }
            }
        };
    }


    //endregion

    @SuppressWarnings("NullableProblems")
    @Override
    public String toString() {
        return "BannerCache{" +
                "configuration=" + configuration +
                '}';
    }

    /**
     * Optional delegate informing about events in BannerCache.
     */
    public interface CacheDelegate {
        /**
         * Called when the first banner gets loaded for the cache. Only called once.
         */
        void firstBannerLoaded();
    }
}
