package com.intentsoftware.addapptr;

import android.util.Log;

import com.intentsoftware.addapptr.module.Logger;

/**
 * Used to set up {@link BannerCache}.
 */
public class BannerCacheConfiguration {
    private static final int MAX_CACHE_SIZE = 5;

    private final String placementName;
    private final int size;
    private BannerCache.CacheDelegate delegate;
    private boolean shouldCacheAdditionalAdAtStart;
    private BannerRequest requestConfiguration = new BannerRequest(null); //prepare default if user does not need extra configuration
    private int minDelayMillis = 1000;

    /**
     * Creates the configuration object to be used when creating BannerCache
     *
     * @param placementName The name of the banner placement that will be created. <b>The placement will be created by the cache and should not be created manually.</b>
     * @param size          Defines how many preloaded banners should be available in the cache. Should be smaller than {@value MAX_CACHE_SIZE}.
     */
    public BannerCacheConfiguration(String placementName, int size) {
        this.placementName = placementName;
        if (size <= 0) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "Cache size cannot be lower than 1! Size = 1 will be used.");
            }
            size = 1;
        } else if (size > MAX_CACHE_SIZE) {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "Cache size cannot be greater than " + MAX_CACHE_SIZE + ". Size = " + MAX_CACHE_SIZE + " will be used.");
            }
            size = MAX_CACHE_SIZE;
        }
        this.size = size;
    }

    BannerCacheConfiguration(BannerCacheConfiguration configuration) {
        this.placementName = configuration.placementName;
        this.size = configuration.size;
        this.delegate = configuration.delegate;
        this.shouldCacheAdditionalAdAtStart = configuration.shouldCacheAdditionalAdAtStart;
        this.requestConfiguration = configuration.requestConfiguration;
        this.minDelayMillis = configuration.minDelayMillis;
    }

    /**
     * Sets optional cache delegate.
     *
     * @param delegate {@link BannerCache.CacheDelegate} to be used.
     */
    public void setDelegate(BannerCache.CacheDelegate delegate) {
        this.delegate = delegate;
    }

    /**
     * Defines if the cache should load additional ad at the beginning.
     *
     * @param shouldCacheAdditionalAdAtStart Boolean indicating if additional ad should be cached.
     */
    public void setShouldCacheAdditionalAdAtStart(boolean shouldCacheAdditionalAdAtStart) {
        this.shouldCacheAdditionalAdAtStart = shouldCacheAdditionalAdAtStart;
    }

    /**
     * The configuration that will be used when requesting new banners.
     *
     * @param requestConfiguration The {@link BannerRequest} instance.
     */
    public void setRequestConfiguration(BannerRequest requestConfiguration) {
        this.requestConfiguration = requestConfiguration;
    }

    /**
     * Sets the minimum delay between two banner consumptions. 1s by default.
     *
     * @param minDelay The minimum delay in seconds between two banner consumptions.
     */
    public void setMinimumDelay(int minDelay) {
        this.minDelayMillis = minDelay * 1000;
    }

    boolean shouldCacheAdditionalAdAtStart() {
        return shouldCacheAdditionalAdAtStart;
    }

    BannerCache.CacheDelegate getDelegate() {
        return delegate;
    }

    int getSize() {
        return size;
    }

    String getPlacementName() {
        return placementName;
    }

    BannerRequest getRequestConfiguration() {
        return requestConfiguration;
    }

    int getMinDelayMillis() {
        return minDelayMillis;
    }

    @Override
    public String toString() {
        return "BannerCacheConfiguration{" +
                "placementName='" + placementName + '\'' +
                ", size=" + size +
                ", delegate=" + delegate +
                ", shouldCacheAdditionalAdAtStart=" + shouldCacheAdditionalAdAtStart +
                ", requestConfiguration=" + requestConfiguration +
                ", minDelayMillis=" + minDelayMillis +
                '}';
    }
}
