package com.intentsoftware.addapptr;

import android.app.Activity;
import android.content.Context;
import android.util.Log;

import com.intentsoftware.addapptr.consent.CMP;
import com.intentsoftware.addapptr.consent.CMPDelegate;
import com.intentsoftware.addapptr.module.Logger;

import androidx.annotation.NonNull;

/**
 * Will present a dialog for the users to grant or withhold their consent.
 */
@SuppressWarnings({"unused", "WeakerAccess"})
public class ManagedConsent extends ConsentImplementation implements CMPDelegate {

    private CMPImplementation cmp;
    private ManagedConsentDelegate delegate;
    private Context applicationContext;

    /**
     * Creates the {@link ManagedConsent} instance.
     *
     * @param cmp      Instance of {@link CMP} to be used.
     * @param context  The {@link Context} of your application.
     * @param delegate The {@link ManagedConsentDelegate} instance. Must not be null.
     */
    public ManagedConsent(CMP cmp, Context context, @NonNull ManagedConsentDelegate delegate) {
        if (cmp instanceof CMPImplementation) {
            if (((CMPImplementation) cmp).isSuccessfullyInitialized()) {
                this.cmp = (CMPImplementation) cmp;
                this.cmp.setDelegate(this);
                this.applicationContext = context.getApplicationContext();
                this.delegate = delegate;
                if (Logger.isLoggable(Log.VERBOSE)) {
                	Logger.v(this, "initialized ManagedConsent: " + toString());
                }
            } else {
                if (Logger.isLoggable(Log.ERROR)) {
                	Logger.e(this, "Passed CMP was not successfully initialized. ManagedConsent will not work.");
                }
            }
        } else {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "Passed CMP is not an instance of allowed classes. ManagedConsent will not work.");
            }
        }
    }

    /**
     * Presents the consent screen ONLY if it is required by the used CMP (i.e. no user consent has been set yet).
     *
     * @param activity The Activity from which the consent screen will be shown.
     */
    public void showIfNeeded(Activity activity) {
        if (Logger.isLoggable(Log.VERBOSE)) {
        	Logger.v(this, "showIfNeeded called");
        }
        if (cmp != null && cmp.isSuccessfullyInitialized()) {
            cmp.showIfNeeded(activity);
        } else {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "ManagedConsent was not initialized with a working CMP. Cannot show.");
            }
        }
    }

    /**
     * Presents the consent screen, allowing the user to change consent settings.
     *
     * @param activity The Activity from which the consent screen will be shown.
     */
    public void editConsent(Activity activity) {
        if (Logger.isLoggable(Log.VERBOSE)) {
        	Logger.v(this, "editConsent called");
        }
        if (cmp != null && cmp.isSuccessfullyInitialized()) {
            cmp.editConsent(activity);
        } else {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "ManagedConsent was not initialized with a working CMP. Cannot edit.");
            }
        }
    }

    /**
     * Tells the CMP to reload. You can call this method for example after receiving {@link ManagedConsentDelegate#managedConsentCMPFailedToLoad(ManagedConsent, String)} callback.
     *
     * @param activity The Activity instance.
     */
    public void reload(Activity activity) {
        if (Logger.isLoggable(Log.VERBOSE)) {
        	Logger.v(this, "reload called");
        }
        if (cmp != null && cmp.isSuccessfullyInitialized()) {
            cmp.reload(activity);
        } else {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "ManagedConsent was not initialized with a working CMP. Cannot reload.");
            }
        }
    }

    @Override
    public void onConsentUpdated(ManagedConsentState state) {
        readConsentStringFromSharedPreferences(applicationContext);
        ConsentHelper.reconfigureNetworks(applicationContext);
        delegate.managedConsentCMPFinished(state);
    }

    @Override
    public void onCMPFailedToShow(String error) {
        if (Logger.isLoggable(Log.WARN)) {
            Logger.w(this, "Failed to show CMP: " + error);
        }
        delegate.managedConsentCMPFailedToShow(this, error);
    }

    @Override
    public void onCMPFailedToLoad(String error) {
        if (Logger.isLoggable(Log.WARN)) {
            Logger.w(this, "Failed to load CMP: " + error);
        }
        delegate.managedConsentCMPFailedToLoad(this, error);
    }

    @Override
    public void CMPNeedsUI() {
        if (Logger.isLoggable(Log.VERBOSE)) {
            Logger.v(this, "CMP needs UI");
        }
        delegate.managedConsentNeedsUserInterface(this);
    }

    @Override
    NonIABConsent getConsentForNetwork(AdNetwork network) {
        if (cmp != null && cmp.isSuccessfullyInitialized()) {
            return cmp.getConsentForNetwork(network);
        } else {
            if (Logger.isLoggable(Log.ERROR)) {
                Logger.e(this, "ManagedConsent was not initialized with a working CMP, cannot check consent for network");
            }
            return NonIABConsent.WITHHELD;
        }
    }

    @Override
    public String toString() {
        return "ManagedConsent{" +
                "cmp=" + cmp +
                ", delegate=" + delegate +
                "} " + super.toString();
    }

    /**
     * Notifies about the need to show the consent dialog.
     */
    public interface ManagedConsentDelegate {
        /**
         * Notifies that {@link ManagedConsent} needs to show a consent dialog. After receiving this notification, you should pause your application and call the {@link #showIfNeeded(Activity)} method.
         *
         * @param managedConsent The {@link ManagedConsent} instance that requests UI.
         */
        void managedConsentNeedsUserInterface(ManagedConsent managedConsent);

        /**
         * Notifies that the used {@link CMP} has finished updating consent.
         * @param state The {@link ManagedConsentState} informing about consent given by the user.
         */
        void managedConsentCMPFinished(ManagedConsentState state);

        /**
         * Notifies that the used {@link CMP} failed to load.
         *
         * @param managedConsent @param managedConsent The {@link ManagedConsent} instance.
         * @param error          The description of what went wrong
         */
        void managedConsentCMPFailedToLoad(ManagedConsent managedConsent, String error);

        /**
         * Notifies that the used {@link CMP} failed to show.
         *
         * @param managedConsent @param managedConsent The {@link ManagedConsent} instance.
         * @param error          The description of what went wrong
         */
        void managedConsentCMPFailedToShow(ManagedConsent managedConsent, String error);
    }

    /**
     * Possible states of consent given by the user.
     */
    public enum ManagedConsentState {
        /**
         * No information about consent state.
         */
        UNKNOWN,

        /**
         * Consent has been declined by the user.
         */
        WITHHELD,

        /**
         * Partial consent has been granted by the user - at least some purposes and some vendors were given consent.
         */
        CUSTOM,

        /**
         * Full consent has been granted by the user.
         */
        OBTAINED
    }

}
