package com.intentsoftware.addapptr.consent;

import android.app.Activity;
import android.content.Context;
import android.preference.PreferenceManager;
import android.util.Log;

import com.google.android.ump.ConsentForm;
import com.google.android.ump.ConsentInformation;
import com.google.android.ump.ConsentRequestParameters;
import com.google.android.ump.FormError;
import com.google.android.ump.UserMessagingPlatform;
import com.intentsoftware.addapptr.AdNetwork;
import com.intentsoftware.addapptr.CMPImplementation;
import com.intentsoftware.addapptr.ManagedConsent;
import com.intentsoftware.addapptr.NonIABConsent;
import com.intentsoftware.addapptr.module.Logger;

import java.util.Arrays;
import java.util.List;

import androidx.annotation.Nullable;

@SuppressWarnings("unused")
public class CMPGoogle extends CMPImplementation {

    private enum MissedCallback {
        FAILED_TO_LOAD,
        FAILED_TO_SHOW,
        NEEDS_UI,
        CONSENT_UPDATED
    }

    private static final String MORE_PARTNERS_STRING_KEY = "IABTCF_AddtlConsent";

    private CMPDelegate delegate;

    private ConsentInformation consentInformation;
    private ConsentForm consentForm;
    private boolean needsUI;
    private List<String> enabledAdditionalPartners;

    private boolean consentInfoLoading;
    private boolean consentFormLoading;
    private MissedCallback missedCallback; //to handle Google notifications sent before delegate is set
    private String missedCallbackErrorDescription;

    /**
     * Constructor.
     *
     * @param activity your Activity.
     */
    public CMPGoogle(final Activity activity) {
        if (checkRequiredClasses("com.google.android.ump.ConsentForm", "com.google.android.ump.ConsentInformation",
                "com.google.android.ump.ConsentRequestParameters", "com.google.android.ump.FormError", "com.google.android.ump.UserMessagingPlatform")) {
            updateMorePartnersData(activity);
            loadConsentInfo(activity);
            onSuccessfulInitialization();
        }
    }

    protected void setDelegate(CMPDelegate delegate) {
        this.delegate = delegate;
        if (missedCallback != null) {
            handleMissedCallback();
        }
    }

    private void handleMissedCallback() {
        switch (missedCallback) {
            case FAILED_TO_LOAD:
                delegate.onCMPFailedToLoad(missedCallbackErrorDescription);
                break;
            case FAILED_TO_SHOW:
                delegate.onCMPFailedToShow(missedCallbackErrorDescription);
                break;
            case NEEDS_UI:
                delegate.CMPNeedsUI();
                break;
            case CONSENT_UPDATED:
                delegate.onConsentUpdated(ManagedConsent.ManagedConsentState.UNKNOWN); //this should actually never happen, so we are good just leaving it as UNKNOWN
                break;
        }
        missedCallbackErrorDescription = null;
        missedCallback = null;
    }

    private void loadConsentInfo(final Activity activity) {
        if (consentInfoLoading) {
            if (Logger.isLoggable(Log.VERBOSE)) {
                Logger.v(this, "Consent info already loading, reload request ignored");
            }
            return;
        }
        if (Logger.isLoggable(Log.VERBOSE)) {
            Logger.v(this, "Loading consent info");
        }

        ConsentRequestParameters params = new ConsentRequestParameters.Builder().build();
        consentInfoLoading = true;

        final ConsentInformation consentInformation = UserMessagingPlatform.getConsentInformation(activity);
        consentInformation.requestConsentInfoUpdate(activity, params,
                new ConsentInformation.OnConsentInfoUpdateSuccessListener() {
                    @Override
                    public void onConsentInfoUpdateSuccess() {
                        if (Logger.isLoggable(Log.VERBOSE)) {
                            Logger.v(CMPGoogle.this, "Consent info update successful");
                        }
                        consentInfoLoading = false;
                        if (consentInformation.isConsentFormAvailable()) {
                            CMPGoogle.this.consentInformation = consentInformation;
                            loadForm(activity);
                        } else {
                            if (Logger.isLoggable(Log.WARN)) {
                            	Logger.w(CMPGoogle.this, "Consent form not available");
                            }
                            String description = "Consent form not available";
                            if (delegate != null) {
                                delegate.onCMPFailedToLoad(description);
                            } else {
                                missedCallback = MissedCallback.FAILED_TO_LOAD;
                                missedCallbackErrorDescription = description;
                            }
                        }
                    }
                },
                new ConsentInformation.OnConsentInfoUpdateFailureListener() {
                    @Override
                    public void onConsentInfoUpdateFailure(FormError formError) {
                        consentInfoLoading = false;
                        String description = "Failed to update consent info: " + formError.getMessage();
                        if (delegate != null) {
                            delegate.onCMPFailedToLoad(description);
                        } else {
                            missedCallback = MissedCallback.FAILED_TO_LOAD;
                            missedCallbackErrorDescription = description;
                        }
                    }
                });
    }

    private void loadForm(Context context) {
        if (consentFormLoading) {
            if (Logger.isLoggable(Log.VERBOSE)) {
                Logger.v(this, "Consent form already loading, reload request ignored");
            }
            return;
        }
        if (Logger.isLoggable(Log.VERBOSE)) {
            Logger.v(this, "Loading consent form");
        }
        consentFormLoading = true;

        UserMessagingPlatform.loadConsentForm(context,
                new UserMessagingPlatform.OnConsentFormLoadSuccessListener() {
                    @Override
                    public void onConsentFormLoadSuccess(ConsentForm consentForm) {
                        if (Logger.isLoggable(Log.VERBOSE)) {
                            Logger.v(CMPGoogle.this, "Consent form loaded");
                        }
                        consentFormLoading = false;
                        CMPGoogle.this.consentForm = consentForm;
                        if (consentInformation.getConsentStatus() == ConsentInformation.ConsentStatus.REQUIRED) {
                            needsUI = true;
                            if (delegate != null) {
                                delegate.CMPNeedsUI();
                            } else {
                                missedCallback = MissedCallback.NEEDS_UI;
                            }
                        }
                    }
                },
                new UserMessagingPlatform.OnConsentFormLoadFailureListener() {
                    @Override
                    public void onConsentFormLoadFailure(FormError formError) {
                        consentFormLoading = false;
                        String description = "Failed to load consent form: " + formError.getMessage();
                        if (delegate != null) {
                            delegate.onCMPFailedToLoad(description);
                        } else {
                            missedCallback = MissedCallback.FAILED_TO_LOAD;
                            missedCallbackErrorDescription = description;
                        }
                    }
                }
        );
    }

    @Override
    protected void reload(Activity activity) {
        if (this.consentInformation == null) {
            loadConsentInfo(activity);
        } else if (this.consentForm == null) {
            loadForm(activity);
        }
    }

    @Override
    protected void showIfNeeded(Activity activity) {
        if (needsUI) {
            editConsent(activity);
        }
    }

    @Override
    protected void editConsent(final Activity activity) {
        if (consentForm != null) {
            needsUI = false;
            consentForm.show(activity,
                    new ConsentForm.OnConsentFormDismissedListener() {
                        @Override
                        public void onConsentFormDismissed(@Nullable FormError formError) {
                            // Handle dismissal by reloading form.
                            updateMorePartnersData(activity);
                            CMPGoogle.this.consentForm = null;
                            loadForm(activity);
                            if (delegate != null) {
                                ManagedConsent.ManagedConsentState state;
                                String vendorsConsentString = PreferenceManager.getDefaultSharedPreferences(activity).getString("IABTCF_VendorConsents", null);
                                String purposesConsentString = PreferenceManager.getDefaultSharedPreferences(activity).getString("IABTCF_PurposeConsents", null);
                                if (vendorsConsentString == null || purposesConsentString == null) {
                                    if (Logger.isLoggable(Log.ERROR)) {
                                    	Logger.e(this, "At least one of the required consent strings was not saved!" +
                                                "\nvendor consents: " + vendorsConsentString +
                                                "\npurpose consents:" + purposesConsentString);
                                    }
                                    state = ManagedConsent.ManagedConsentState.UNKNOWN;
                                } else if (vendorsConsentString.contains("1") && purposesConsentString.contains("1")) { //at least one vendor and one purpose enabled
                                    state = ManagedConsent.ManagedConsentState.CUSTOM;
                                } else {
                                    state = ManagedConsent.ManagedConsentState.WITHHELD;
                                }

                                delegate.onConsentUpdated(state);
                            } else {
                                missedCallback = MissedCallback.CONSENT_UPDATED;
                            }
                        }
                    });
        } else {
            String description = "Consent form not ready";
            if (delegate != null) {
                delegate.onCMPFailedToShow(description);
            } else {
                missedCallback = MissedCallback.FAILED_TO_SHOW;
                missedCallbackErrorDescription = description;
            }
        }
    }

    @Override
    protected NonIABConsent getConsentForNetwork(AdNetwork network) {
        switch (network) {
            case APPLOVIN:
                return readConsentForProviderId(1301);
            case FACEBOOK:
                return readConsentForProviderId(89);
            case ADMOB:
            case ADX:
            case DFP:
                return readConsentForProviderId(229);
            case PUBNATIVE:
                return readConsentForProviderId(2977);
            case YANDEX:
                return readConsentForProviderId(1033);
            case MOPUB:
                return readConsentForProviderId(1031);
            default:
                //FIXME: missing mappings for Huawei, InMobi, UnityAds
                if (Logger.isLoggable(Log.WARN)) {
                    Logger.w(this, "No mapping for network " + network + " available, treating consent as withheld");
                }
                return NonIABConsent.WITHHELD;
        }
    }

    private void updateMorePartnersData(Context context) {
        String morePartnersConsentString = PreferenceManager.getDefaultSharedPreferences(context).getString(MORE_PARTNERS_STRING_KEY, null);
        enabledAdditionalPartners = null;
        if (morePartnersConsentString != null) {
            morePartnersConsentString = morePartnersConsentString.replaceFirst(".*~", ""); //remove "1~" or similar
            String[] partnersArray = morePartnersConsentString.split("\\.");
            enabledAdditionalPartners = Arrays.asList(partnersArray);
        } else {
            if (Logger.isLoggable(Log.VERBOSE)) {
                Logger.v(this, "No more partners string found");
            }
        }
    }

    private NonIABConsent readConsentForProviderId(int id) {
        if (enabledAdditionalPartners != null) {
            if (enabledAdditionalPartners.contains(String.valueOf(id))) {
                return NonIABConsent.OBTAINED;
            } else {
                return NonIABConsent.WITHHELD;
            }
        } else {
            if (Logger.isLoggable(Log.WARN)) {
                Logger.w(this, "Enabled additional partners list is not available");
            }
            return NonIABConsent.WITHHELD;
        }
    }
}
