package com.intentsoftware.addapptr.consent;

import android.app.Activity;
import android.app.Application;
import android.util.Log;

import com.intentsoftware.addapptr.AdNetwork;
import com.intentsoftware.addapptr.CMPImplementation;
import com.intentsoftware.addapptr.ManagedConsent;
import com.intentsoftware.addapptr.NonIABConsent;
import com.intentsoftware.addapptr.ad.networkhelpers.OguryHelper;
import com.intentsoftware.addapptr.module.Logger;
import com.ogury.cm.OguryChoiceManager;
import com.ogury.cm.OguryConsentListener;
import com.ogury.core.OguryError;

public class CMPOgury extends CMPImplementation {

    private CMPDelegate delegate;
    private boolean asked;
    private OguryConsentListener oguryListener;
    private String initializationError;

    /**
     * Constructor.
     *
     * @param application the Application context.
     * @param assetKey    the Asset key of your application.
     */
    public CMPOgury(Application application, String assetKey) {
        if (checkRequiredClasses("com.ogury.sdk.Ogury", "com.ogury.cm.OguryChoiceManager", "com.ogury.cm.OguryConsentListener", "com.ogury.core.OguryError")) {
            try {
                OguryHelper.initOgurySDK(application, assetKey);
                oguryListener = new OguryConsentListener() {
                    @Override
                    public void onComplete(OguryChoiceManager.Answer answer) {
                        if (delegate != null) {
                            ManagedConsent.ManagedConsentState state;
                            switch (answer) {
                                case FULL_APPROVAL:
                                    state = ManagedConsent.ManagedConsentState.OBTAINED;
                                    break;
                                case PARTIAL_APPROVAL:
                                    state = ManagedConsent.ManagedConsentState.CUSTOM;
                                    break;
                                case REFUSAL:
                                    state = ManagedConsent.ManagedConsentState.WITHHELD;
                                    break;
                                default:
                                    state = ManagedConsent.ManagedConsentState.UNKNOWN;
                            }

                            delegate.onConsentUpdated(state);
                        }
                    }

                    @Override
                    public void onError(OguryError oguryError) {
                        if (delegate != null) {
                            delegate.onCMPFailedToShow(oguryError.getMessage());
                        }
                    }
                };
                onSuccessfulInitialization();
            } catch (Exception e) {
                if (delegate != null) {
                    delegate.onCMPFailedToLoad(e.getMessage());
                } else {
                    initializationError = e.getMessage();
                }

                if (Logger.isLoggable(Log.ERROR)) {
                    Logger.e(this, e.getMessage());
                }
            }
        }
    }

    @Override
    protected void showIfNeeded(Activity activity) {
        OguryChoiceManager.ask(activity, oguryListener);
        asked = true;
    }

    @Override
    protected void editConsent(Activity activity) {
        OguryChoiceManager.edit(activity, oguryListener);
    }

    @Override
    protected void setDelegate(CMPDelegate delegate) {
        this.delegate = delegate;
        if (initializationError != null) { //handle init failing before delegate is set
            delegate.onCMPFailedToLoad(initializationError);
            initializationError = null;
        }
    }

    @Override
    protected NonIABConsent getConsentForNetwork(AdNetwork network) {
        if (!asked) {
            return NonIABConsent.WITHHELD; //handle checks before CMP is asked for consent status
        }

        switch (network) {
            case APPLOVIN:
                return getCosentForVendorId(5);
            case FACEBOOK:
                return getCosentForVendorId(9);
            case ADMOB:
            case ADX:
            case DFP:
                return getCosentForVendorId(3);
            case INMOBI:
                return getCosentForVendorId(296);
            case MOPUB:
                return getCosentForVendorId(12);
            case UNITYADS:
                return getCosentForVendorId(16);
            case PUBNATIVE:
                return getCosentForVendorId(139);
            default:
                // TODO: no mappings for Huawei and Yandex
                if (Logger.isLoggable(Log.WARN)) {
                    Logger.w(this, "No mapping for network " + network + " available, treating consent as withheld");
                }
                return NonIABConsent.WITHHELD;
        }
    }

    private NonIABConsent getCosentForVendorId(int vendorId) {
        return OguryChoiceManager.TcfV2.isAccepted(vendorId) ? NonIABConsent.OBTAINED : NonIABConsent.WITHHELD;
    }

    @Override
    public void reload(Activity activity) {
    }
}
