@file:Suppress("unused")

package com.intentsoftware.addapptr

import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.util.Log
import android.util.Pair
import android.view.View
import android.view.ViewGroup
import com.intentsoftware.addapptr.internal.ad.NativeAd.NativeAdRating
import com.intentsoftware.addapptr.ad.NativeAdData
import com.intentsoftware.addapptr.ad.VASTAdData
import com.intentsoftware.addapptr.internal.*
import com.intentsoftware.addapptr.internal.AdController
import com.intentsoftware.addapptr.internal.BannerReloader
import com.intentsoftware.addapptr.internal.module.Logger.d
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.isLoggable
import com.intentsoftware.addapptr.internal.module.Logger.v
import com.intentsoftware.addapptr.internal.module.Logger.w
import com.intentsoftware.addapptr.internal.module.ServerLogger
import com.intentsoftware.addapptr.internal.module.ServerLogger.log
import com.intentsoftware.addapptr.internal.module.Utils
import com.intentsoftware.addapptr.internal.module.Utils.fittingBannerSizes
import com.intentsoftware.addapptr.internal.module.Utils.maxPlacementSize
import com.intentsoftware.addapptr.internal.module.Utils.screenWidthLandscape
import com.intentsoftware.addapptr.internal.module.Utils.screenWidthPortrait

//TODO: use null-safe calls (?.) ?

object AATKit {
    // Used only in Javadoc.
    const val BANNER_DEFAULT_RELOAD_INTERVAL_IN_SECONDS =
        (BannerReloader.AUTORELOAD_INTERVAL_DEFAULT / 1000).toInt()
    const val BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS =
        (BannerReloader.AUTORELOAD_INTERVAL_MIN / 1000).toInt()
    const val BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS =
        (BannerReloader.AUTORELOAD_INTERVAL_MAX / 1000).toInt()
    private var adController: AdController? = null

    /**
     * @return AATKit version.
     */
    @JvmStatic
    val version: String
        get() {
            logAATKitCall("CMD: getVersion()")
            return Version.NAME
        }

    /**
     * @return AATKit detailed build version.
     */
    @JvmStatic
    val fullVersion: String
        get() {
            logAATKitCall("CMD: getFullVersion()")
            return Version.FULL_NAME
        }

    /**
     * Initializes the AATKit library. Should be called once during application initialization before any other calls to AATKit.
     *
     * @param configuration Configuration for AATKit.
     * @see AATKitConfiguration
     */
    @JvmStatic
    fun init(configuration: AATKitConfiguration) {
        logAATKitCall("CMD: init($configuration)")
        if (adController != null) {
            if (isLoggable(Log.ERROR)) {
                e(AATKit::class.java, "AdController is already initialized")
            }
            return
        }
        var version = "?"
        val packageName = configuration.application.applicationContext.packageName
        try {
            version =
                configuration.application.packageManager.getPackageInfo(packageName, 0).versionName
        } catch (e: PackageManager.NameNotFoundException) {
            if (isLoggable(Log.WARN)) {
                w(AATKit::class.java, "Failed to check version of application", e)
            }
        }
        d(
            "Init", Version.FULL_NAME
                    + ", application: " + packageName
                    + " (version: " + version + ")"
                    + ", configuration: " + configuration
        )
        adController = AdController(configuration)
    }

    /**
     * Enables debug screen that will show after shaking the device. It is already enabled by default.
     */
    @JvmStatic
    fun enableDebugScreen() {
        logAATKitCall("CMD: enableDebugScreen()")
        adController!!.enableDebugScreen()
    }

    /**
     * Disables the debug screen appearing after shaking the device. It is enabled by default.
     */
    @JvmStatic
    fun disableDebugScreen() {
        logAATKitCall("CMD: disableDebugScreen()")
        adController!!.disableDebugScreen()
    }

    /**
     * Used for obtaining debug information (the same that would be presented in dialog after shaking the device if debug screen is enabled)
     *
     * @return String with debug information
     */
    @JvmStatic
    val debugInfo: String
        get() {
            logAATKitCall("CMD: getDebugInfo()")
            return adController!!.debugInfo
        }

    /**
     * Allows to reconfigure the options for GDPR consent.
     *
     * @param configuration New configuration.
     */
    @JvmStatic
    fun reconfigure(configuration: AATKitRuntimeConfiguration) {
        logAATKitCall("CMD: reconfigure($configuration)")
        adController!!.reconfigure(configuration)
    }

    /**
     * Checks if AATKit recognizes given device as tablet.
     *
     * @param context The [Context] of your application.
     * @return True if device is recognized as tablet, false otherwise.
     */
    @JvmStatic
    fun isTablet(context: Context): Boolean {
        logAATKitCall("CMD: isTablet($context)")
        return Utils.isTablet(context)
    }

    /**
     * Returns the [PlacementSize] with maximum width that will fit on a given device in portrait screen orientation.
     *
     * @param context The [Context] of your application.
     * @return [PlacementSize] best fitting current device
     */
    @JvmStatic
    fun maximumBannerSizePortrait(context: Context): PlacementSize {
        logAATKitCall("CMD: maximumBannerSizePortrait($context)")
        return maxPlacementSize(screenWidthPortrait(context))
    }

    /**
     * Returns the [PlacementSize] with maximum width that will fit on a given device in landscape screen orientation.
     *
     * @param context The [Context] of your application.
     * @return [PlacementSize] best fitting current device
     */
    @JvmStatic
    fun maximumBannerSizeLandscape(context: Context): PlacementSize {
        logAATKitCall("CMD: maximumBannerSizeLandscape($context)")
        return maxPlacementSize(screenWidthLandscape(context))
    }

    /**
     * Returns the set of [BannerSize] that will fit on a given device in portrait screen orientation.
     *
     * @param context The [Context] of your application.
     * @return Set of [BannerSize] fitting current device
     */
    @JvmStatic
    fun fittingBannerSizesPortrait(context: Context): Set<BannerSize> {
        logAATKitCall("CMD: fittingBannerSizesPortrait($context)")
        return fittingBannerSizes(screenWidthPortrait(context))
    }

    /**
     * Returns the set of [BannerSize] that will fit on a given device in landscape screen orientation.
     *
     * @param context The [Context] of your application.
     * @return Set of [BannerSize] fitting current device
     */
    @JvmStatic
    fun fittingBannerSizesLandscape(context: Context): Set<BannerSize> {
        logAATKitCall("CMD: fittingBannerSizesLandscape($context)")
        return fittingBannerSizes(screenWidthLandscape(context))
    }

    /**
     * Notifies AATKit about activity resume. Invoke this method in every activity that uses AATKit.
     *
     * @param activity Reference to [Activity].
     */
    @JvmStatic
    fun onActivityResume(activity: Activity) {
        logAATKitCall("CMD: onActivityResume($activity)")
        adController!!.onActivityResume(activity)
    }

    /**
     * Notifies AATKit about activity pause. Invoke this method in every activity that uses AATKit.
     *
     * @param activity Reference to [Activity].
     */
    @JvmStatic
    fun onActivityPause(activity: Activity) {
        logAATKitCall("CMD: onActivityPause($activity)")
        adController!!.onActivityPause()
    }

    /**
     * Creates placement with given name and size. If the placement of given name and size already exists, it will be returned.
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @param size Size of placement. Use [PlacementSize.Fullscreen] for interstitials, [PlacementSize.Native] for native ads and other sizes for banner ads.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createPlacement(name: String, size: PlacementSize): Int {
        logAATKitCall("CMD: createPlacement($name,$size)")
        return adController!!.createPlacement(name, size, null)
    }

    /**
     * Creates placement with given name and size. If the placement of given name and size already exists, it will be returned.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @param size               Size of placement. Use [PlacementSize.Fullscreen] for interstitials, [PlacementSize.Native] for native ads and other sizes for banner ads.
     * @param statisticsListener [AATKit.StatisticsListener] implementation that will be notified about statistics events.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createPlacement(
        name: String,
        size: PlacementSize,
        statisticsListener: StatisticsListener?
    ): Int {
        logAATKitCall("CMD: createPlacement($name,$size,$statisticsListener)")
        return adController!!.createPlacement(name, size, statisticsListener)
    }

    /**
     * Creates a new native ad placement. If the native ad placement of given name already exists, it will be returned.
     *
     * @param name              Unique name of placement. The same name will be used in addapptr.com account.
     * @param supportsMainImage True if the native ads returned should have main image asset. Keep in mind that if main image is used, it has to be displayed.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createNativeAdPlacement(name: String, supportsMainImage: Boolean): Int {
        logAATKitCall("CMD: createNativeAdPlacement($name,$supportsMainImage)")
        return adController!!.createNativeAdPlacement(name, supportsMainImage, null)
    }

    /**
     * Creates a new native ad placement. If the native ad placement of given name already exists, it will be returned.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @param supportsMainImage  True if the native ads returned should have main image asset. Keep in mind that if main image is used, it has to be displayed.
     * @param statisticsListener [AATKit.StatisticsListener] implementation that will be notified about statistics events.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createNativeAdPlacement(
        name: String,
        supportsMainImage: Boolean,
        statisticsListener: StatisticsListener?
    ): Int {
        logAATKitCall("CMD: createNativeAdPlacement($name,$supportsMainImage,$statisticsListener)")
        return adController!!.createNativeAdPlacement(name, supportsMainImage, statisticsListener)
    }

    /**
     * Creates a new banner placement. If the banner placement of given name already exists, it will be returned.
     *
     *
     * **The placement will create a copy of the
     * configuration.** Any changes made to the configuration after placement is created will be ignored.
     *
     * @param name          Unique name of placement. The same name will be used in addapptr.com account.
     * @param configuration The configuration for this placement. The placement will ignore any changes made to configuration after it was created.
     * @return Banner placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createBannerPlacement(name: String, configuration: BannerConfiguration?): BannerPlacement? {
        logAATKitCall("CMD: createInFeedBannerPlacement($name,$configuration)")
        return adController!!.createBannerPlacement(name, configuration, null)
    }

    /**
     * Creates a new banner placement. If the banner placement of given name already exists, it will be returned.
     *
     *
     * **The placement will create a copy of the
     * configuration.** Any changes made to the configuration after placement is created will be ignored.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @param configuration      The configuration for this placement. The placement will ignore any changes made to configuration after it was created.
     * @param statisticsListener [AATKit.StatisticsListener] implementation that will be notified about statistics events.
     * @return Banner placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createBannerPlacement(
        name: String,
        configuration: BannerConfiguration?,
        statisticsListener: StatisticsListener?
    ): BannerPlacement? {
        logAATKitCall("CMD: createInFeedBannerPlacement($name,$configuration,$statisticsListener)")
        return adController!!.createBannerPlacement(name, configuration, statisticsListener)
    }

    /**
     * Creates a new rewarded video placement. If the rewarded video ad placement of given name already exists, it will be returned.
     * **Only one Rewarded Video placement can be used within the app.**
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createRewardedVideoPlacement(name: String): Int {
        logAATKitCall("CMD: createRewardedVideoPlacement($name)")
        return adController!!.createRewardedVideoPlacement(name, null)
    }

    /**
     * Creates a new rewarded video placement. If the rewarded video ad placement of given name already exists, it will be returned.
     * **Only one Rewarded Video placement can be used within the app.**
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @param statisticsListener [AATKit.StatisticsListener] implementation that will be notified about statistics events.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createRewardedVideoPlacement(name: String, statisticsListener: StatisticsListener?): Int {
        logAATKitCall("CMD: createRewardedVideoPlacement($name,$statisticsListener)")
        return adController!!.createRewardedVideoPlacement(name, statisticsListener)
    }

    /**
     * Creates a new AppOpen ad placement. If the AppOpen ad placement of given name already exists, it will be returned.
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createAppOpenAdPlacement(name: String): Int {
        logAATKitCall("CMD: createAppOpenAdPlacement($name)")
        return adController!!.createAppOpenAdPlacement(name, null)
    }

    /**
     * Creates a new AppOpen ad placement. If the AppOpen ad placement of given name already exists, it will be returned.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @param statisticsListener [AATKit.StatisticsListener] implementation that will be notified about statistics events.
     * @return Placement identifier, or -1 if placement cannot be created.
     */
    @JvmStatic
    fun createAppOpenAdPlacement(name: String, statisticsListener: StatisticsListener?): Int {
        logAATKitCall("CMD: createAppOpenAdPlacement($name)")
        return adController!!.createAppOpenAdPlacement(name, statisticsListener)
    }

    /**
     * Enables automatic reloading of placement. Autoreloader will use reload time configured on addapptr.com account or fallback to default {@value #BANNER_DEFAULT_RELOAD_INTERVAL_IN_SECONDS} seconds.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     */
    @JvmStatic
    fun startPlacementAutoReload(placementId: Int) {
        logAATKitCall("CMD: startPlacementAutoReload($placementId)")
        adController!!.startPlacementAutoReload(placementId)
    }

    /**
     * Enables automatic reloading of placement and sets custom reload time. This reload time will be used instead of time configured on addapptr.com account.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param seconds     Interval between ad reloads. Should be a value from {@value #BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS} to {@value #BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS} seconds.
     */
    @JvmStatic
    fun startPlacementAutoReload(placementId: Int, seconds: Int) {
        logAATKitCall("CMD: startPlacementAutoReload($placementId,$seconds)")
        if (seconds < BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS || seconds > BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS) {
            if (isLoggable(Log.WARN)) {
                w(
                    AATKit::class.java,
                    "Passed autoreload interval: $seconds is not within $BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS to $BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS bounds and will be ignored."
                )
            }
        }
        adController!!.startPlacementAutoReload(placementId, seconds)
    }

    /**
     * Disables automatic reloading of placement.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     */
    @JvmStatic
    fun stopPlacementAutoReload(placementId: Int) {
        logAATKitCall("CMD: stopPlacementAutoReload($placementId)")
        adController!!.stopPlacementAutoReload(placementId)
    }

    /**
     * Sets reload time that will be used instead of time configured on addapptr.com account.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param seconds     Interval between ad reloads. Should be a value from {@value #BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS} to {@value #BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS} seconds.
     */
    @JvmStatic
    fun setPlacementAutoreloadInterval(placementId: Int, seconds: Int) {
        logAATKitCall("CMD: setPlacementAutoreloadInterval($placementId,$seconds)")
        adController!!.setPlacementAutoreloadInterval(placementId, seconds)
    }

    /**
     * Requests placement reload. Works only if automatic reloading is disabled. For native ads, it might return false if current limit for native ads loading in parallel is reached.
     *
     *
     * For banners it will not cause new banner to show more often than set banner refresh interval (default: 30s) - instead, it will set a timer to reload and show next ad when the refresh interval passes. To force ad reload and display before this delay, use the method [.reloadPlacement] instead.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @return True if call ends in causing a placement to reload, false otherwise.
     * @see .reloadPlacement
     */
    @JvmStatic
    fun reloadPlacement(placementId: Int): Boolean {
        logAATKitCall("CMD: reloadPlacement($placementId)")
        return adController!!.reloadPlacement(placementId, false)
    }

    /**
     * Requests placement reload. Works only if automatic reloading is disabled. For native ads, it might return false if current limit for native ads loading in parallel is reached.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param forceLoad   True if you want to be able to reload and show new banner before set banner reload interval (default: 30s) between reloads, false otherwise.
     * @return True if call ends in causing a placement to reload, false otherwise.
     * @see .reloadPlacement
     */
    @JvmStatic
    fun reloadPlacement(placementId: Int, forceLoad: Boolean): Boolean {
        logAATKitCall("CMD: reloadPlacement($placementId,$forceLoad)")
        return adController!!.reloadPlacement(placementId, forceLoad)
    }

    /**
     * Returns placement view. Works only for banner placements.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @return Placement view or null if `placementId` is not valid.
     */
    @JvmStatic
    fun getPlacementView(placementId: Int): View? {
        logAATKitCall("CMD: getPlacementView($placementId)")
        return adController!!.getPlacementView(placementId)
    }

    /**
     * Allows to set the desired position of AdChoices icon in Google native ads. Set null to use the ad network's default.
     *
     * @param position desired AdChoices icon position. Null by default.
     */
    @JvmStatic
    fun setAdChoicesIconPosition(position: AdChoicesIconPosition?) {
        logAATKitCall("CMD: setAdChoicesIconPosition($position)")
        adController!!.setAdChoicesIconPosition(position)
    }

    /**
     * Binds the native ad instance with given ViewGroup. Needed for click handling and tracking.
     *
     * @param nativeAd      Native ad instance.
     * @param layout        ViewGroup used to render the native ad.
     * @param mainImageView View used to show the main image of the ad. Can be null.
     * @param iconView      View used to show the icon of the native ad.
     */
    @JvmStatic
    @Deprecated("This method is deprecated and will be removed in the future. Please use {@link #attachNativeAdToLayout(NativeAdData, ViewGroup, View, View, View)} instead.")
    fun attachNativeAdToLayout(
        nativeAd: NativeAdData,
        layout: ViewGroup,
        mainImageView: View?,
        iconView: View?
    ) {
        val mainImageClass =
            if (mainImageView == null) "null" else mainImageView.javaClass.simpleName
        val iconViewClass = if (iconView == null) "null" else iconView.javaClass.simpleName
        logAATKitCall("CMD: attachNativeAdToLayout(" + nativeAd + ", " + layout.javaClass.simpleName + ", " + mainImageClass + ", " + iconViewClass + ")")
        adController!!.attachNativeAdToLayout(nativeAd, layout, mainImageView, iconView, null)
    }

    /**
     * Binds the native ad instance with given ViewGroup. Needed for click handling and tracking.
     *
     * @param nativeAd      Native ad instance.
     * @param layout        ViewGroup used to render the native ad.
     * @param mainImageView View used to show the main image of the ad. Can be null.
     * @param iconView      View used to show the icon of the native ad.
     * @param CTAView       View used to show the Call To Action of the native ad. Can be null.
     */
    @JvmStatic
    fun attachNativeAdToLayout(
        nativeAd: NativeAdData,
        layout: ViewGroup,
        mainImageView: View?,
        iconView: View?,
        CTAView: View?
    ) {
        val mainImageClass =
            if (mainImageView == null) "null" else mainImageView.javaClass.simpleName
        val iconViewClass = if (iconView == null) "null" else iconView.javaClass.simpleName
        val ctaViewClass = if (CTAView == null) "null" else CTAView.javaClass.simpleName
        logAATKitCall("CMD: attachNativeAdToLayout(" + nativeAd + ", " + layout.javaClass.simpleName + ", " + mainImageClass + ", " + iconViewClass + ", " + ctaViewClass + ")")
        adController!!.attachNativeAdToLayout(nativeAd, layout, mainImageView, iconView, CTAView)
    }

    /**
     * Removes the binding between native ad and ViewGroup. Should be called when the native ad will no longer be presented and should be destroyed.
     *
     * @param nativeAd Native ad instance.
     */
    @JvmStatic
    fun detachNativeAdFromLayout(nativeAd: NativeAdData) {
        logAATKitCall("CMD: detachNativeAdFromLayout($nativeAd)")
        adController!!.detachNativeAdFromLayout(nativeAd)
    }

    /**
     * Returns the title of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with title asset of the ad, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdTitle(nativeAd: NativeAdData): String? {
        logAATKitCall("CMD: getNativeAdTitle($nativeAd)")
        return adController!!.getNativeAdTitle(nativeAd)
    }

    /**
     * Returns the description of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with description asset of the ad, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdDescription(nativeAd: NativeAdData): String? {
        logAATKitCall("CMD: getNativeAdDescription($nativeAd)")
        return adController!!.getNativeAdDescription(nativeAd)
    }

    /**
     * Returns the call to action of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with call to action asset of the ad, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdCallToAction(nativeAd: NativeAdData): String? {
        logAATKitCall("CMD: getNativeAdCallToAction($nativeAd)")
        return adController!!.getNativeAdCallToAction(nativeAd)
    }

    /**
     * Returns the URL of the image asset of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with URL of the image asset of the ad, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdImageUrl(nativeAd: NativeAdData): String? {
        logAATKitCall("CMD: getNativeAdImageUrl($nativeAd)")
        return adController!!.getNativeAdImageUrl(nativeAd)
    }

    /**
     * Returns the URL of the icon asset of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return String with URL of the icon asset of the ad, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdIconUrl(nativeAd: NativeAdData): String? {
        logAATKitCall("CMD: getNativeAdIconUrl($nativeAd)")
        return adController!!.getNativeAdIconUrl(nativeAd)
    }

    /**
     * Returns the rating asset of native ad.
     *
     * @param nativeAd Native ad instance.
     * @return NativeAdRating instance containing both value and scale of rating, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdRating(nativeAd: NativeAdData): NativeAdRating? {
        logAATKitCall("CMD: getNativeAdRating($nativeAd)")
        return adController!!.getNativeAdRating(nativeAd)
    }

    /**
     * Returns the view with branding logo or ad information related to the ad network providing the native ad. Some networks like Facebook Audience Network require this special view to be visible on native ads.
     *
     * @param nativeAd Native ad insance.
     * @return View that should be added to native ad layout, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdBrandingLogo(nativeAd: NativeAdData): View? {
        logAATKitCall("CMD: getNativeAdBrandingLogo($nativeAd)")
        return adController!!.getNativeAdBrandingLogo(nativeAd)
    }

    /**
     * Returns the advertiser asset of native ad (not the same as ad network providing it).
     *
     * @param nativeAd Native ad instance.
     * @return String with advertiser asset of the ad, or null if it is not available.
     */
    @JvmStatic
    fun getNativeAdAdvertiser(nativeAd: NativeAdData): String? {
        logAATKitCall("CMD: getNativeAdAdvertiser($nativeAd)")
        return adController!!.getNativeAdAdvertiser(nativeAd)
    }

    /**
     * Returns the ad network providing given native ad
     *
     * @param nativeAd Native ad instance.
     * @return Enum value representing the ad network providing the ad.
     */
    @JvmStatic
    fun getNativeAdNetwork(nativeAd: NativeAdData): AdNetwork {
        logAATKitCall("CMD: getNativeAdNetwork($nativeAd)")
        return adController!!.getNativeAdNetwork(nativeAd)
    }

    /**
     * Returns if the native ad has expired and shall no longer be used.
     *
     * @param nativeAd Native ad instance.
     * @return True if native ad has expired, false otherwise.
     */
    @JvmStatic
    fun isNativeAdExpired(nativeAd: NativeAdData): Boolean {
        logAATKitCall("CMD: isNativeAdExpired($nativeAd)")
        return adController!!.isNativeAdExpired(nativeAd)
    }

    /**
     * Returns if the native ad is ready to be displayed.
     *
     * @param nativeAd Native ad instance.
     * @return True if native ad is ready, false otherwise.
     */
    @JvmStatic
    fun isNativeAdReady(nativeAd: NativeAdData): Boolean {
        logAATKitCall("CMD: isNativeAdReady($nativeAd)")
        return adController!!.isNativeAdReady(nativeAd)
    }

    /**
     * Indicates that the application wants to display a native or VAST ad for given placement. Used in reporting.
     *
     * @param placementId Placement identifier obtained from
     * [.createPlacement].
     */
    @JvmStatic
    fun reportAdSpaceForPlacement(placementId: Int) {
        logAATKitCall("CMD: reportAdSpaceForPlacement($placementId)")
        adController!!.reportAdSpaceForPlacement(placementId)
    }

    /**
     * Returns how many ads are currently loading for given native ad placement.
     *
     * @param placementId Placement identifier obtained from
     * [.createPlacement].
     * @return Number of ads that are currently loading for given placement.
     */
    @JvmStatic
    fun currentlyLoadingNativeAdsOnPlacement(placementId: Int): Int {
        logAATKitCall("CMD: currentlyLoadingNativeAdsOnPlacement($placementId)")
        return adController!!.currentlyLoadingNativeAdsOnPlacement(placementId)
    }

    /**
     * Checks if desired impression cap (set on AddApptr Dashboard) is reached for given native ad, fullscreen, rewarded video or VAST placement.
     * Note: this method will ALWAYS return "false" for other placement types.
     *
     * @param placementId Placement identifier obtained from
     * [.createPlacement].
     * @return True if impression cap is reached for given placement, false otherwise.
     */
    @JvmStatic
    fun isFrequencyCapReachedForPlacement(placementId: Int): Boolean {
        logAATKitCall("CMD: isFrequencyCapReachedForPlacement($placementId)")
        return adController!!.isFrequencyCapReachedForPlacement(placementId)
    }

    /**
     * Returns the instance of native ad for given native ad placement.
     *
     * @param placementId Placement identifier obtained from
     * [.createPlacement].
     * @return Native ad instance if it is loaded for given placement, null otherwise.
     */
    @JvmStatic
    fun getNativeAd(placementId: Int): NativeAdData? {
        logAATKitCall("CMD: getNativeAd($placementId)")
        return adController!!.getNativeAd(placementId)
    }

    /**
     * Returns identifier of placement with given name, or -1 if placement with given name does not exist.
     *
     * @param placementName Placent name, used when creating placement with [.createPlacement] method.
     * @return Placement identifier, or -1 if placement does not exist.
     */
    @JvmStatic
    fun getPlacementIdForName(placementName: String): Int {
        logAATKitCall("CMD: getPlacementIdForName($placementName)")
        return adController!!.findPlacementIdByRealName(placementName)
    }

    /**
     * Returns true if there is an ad loaded for given placementId.
     *
     * @param placementId Placement identifier obtained from
     * [.createPlacement].
     * @return True if there is an ad loaded for given placementId.
     */
    @JvmStatic
    fun hasAdForPlacement(placementId: Int): Boolean {
        logAATKitCall("CMD: hasAdForPlacement($placementId)")
        return adController!!.hasAdForPlacement(placementId)
    }

    /**
     * Shows interstitial ad if ad is ready.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @return True if showing interstitial was successful, false otherwise.
     */
    @JvmStatic
    fun showPlacement(placementId: Int): Boolean {
        logAATKitCall("CMD: showPlacement($placementId)")
        return adController!!.showPlacement(placementId)
    }

    /**
     * Allows to enable or disable selected ad networks. By default all networks are enabled.
     *
     * @param network Ad network.
     * @param enabled True to enable, false to disable.
     */
    @JvmStatic
    fun setNetworkEnabled(network: AdNetwork, enabled: Boolean) {
        logAATKitCall("CMD: setNetworkEnabled($network,$enabled)")
        SupportedNetworks.setNetworkEnabled(network, enabled)
    }

    /**
     * Returns true if ad network is enabled, false otherwise.
     *
     * @param network Ad network.
     * @return True if ad network is enabled, false otherwise.
     */
    @JvmStatic
    fun isNetworkEnabled(network: AdNetwork): Boolean {
        logAATKitCall("CMD: isNetworkEnabled($network)")
        return SupportedNetworks.isNetworkEnabled(network)
    }

    /**
     * Sets placement default image. This image will be shown in placement when no ad is available.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param image       The bitmap to set.
     */
    @JvmStatic
    fun setPlacementDefaultImageBitmap(placementId: Int, image: Bitmap?) {
        logAATKitCall("CMD: setPlacementDefaultImageBitmap($placementId,$image)")
        adController!!.setPlacementDefaultImage(placementId, image)
    }

    /**
     * Sets placement default image. This image will be shown in placement when no ad is available.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param resId       The identifier of the resource.
     */
    @JvmStatic
    fun setPlacementDefaultImageResource(placementId: Int, resId: Int) {
        logAATKitCall("CMD: setPlacementDefaultImageResource($placementId,$resId)")
        adController!!.setPlacementDefaultImageResource(placementId, resId)
    }

    /**
     * Get option from AATKit. Options can be obtained from the server or set using the [.setOption] method.
     *
     * @param optionName The name of the option to be checked.
     * @return Value of the option or null if it is not set.
     */
    @JvmStatic
    fun getOption(optionName: String): String? {
        logAATKitCall("CMD: getOption($optionName)")
        return AdController.getOption(optionName)
    }

    /**
     * Convenience method for checking if option is enabled in AATKit. Options can be obtained from the server or set using the [.setOption] method.
     *
     * @param optionName The name of the option to be checked.
     * @return True if option value is "Yes", false otherwise.
     */
    @JvmStatic
    fun isOptionEnabled(optionName: String): Boolean {
        logAATKitCall("CMD: isOptionEnabled($optionName)")
        return AdController.isOptionEnabled(optionName)
    }

    /**
     * Set option in AATKit. Options can also be obtained from the server.
     *
     * @param optionName  The name of the option to be set.
     * @param optionValue The value of the option to be set.
     */
    @JvmStatic
    fun setOption(optionName: String, optionValue: String) {
        logAATKitCall("CMD: setOption($optionName,$optionName)")
        adController!!.setOption(optionName, optionValue)
    }

    /**
     * Sets the impression listener for the given placement.
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param impressionListener [ImpressionListener] implementation that will be notified about impression events.
     */
    @JvmStatic
    fun setImpressionListener(placementId: Int, impressionListener: ImpressionListener?) {
        logAATKitCall("CMD: setImpressionListener($placementId,$impressionListener)")
        adController!!.setImpressionListener(placementId, impressionListener)
    }

    /**
     * Sets the impression listener for the given banner placement.
     * @param bannerPlacement [BannerPlacement] instance.
     * @param impressionListener [ImpressionListener] implementation that will be notified about impression events.
     */
    @JvmStatic
    fun setImpressionListener(
        bannerPlacement: BannerPlacement,
        impressionListener: ImpressionListener?
    ) {
        logAATKitCall("CMD: setImpressionListener($bannerPlacement,$impressionListener)")
        adController!!.setImpressionListener(bannerPlacement, impressionListener)
    }

    /**
     * Sets the targeting information for the application. This information will be used only if no placement-specific targeting is available.
     *
     * @param info Map with targeting information
     * @see .setTargetingInfo
     */
    @JvmStatic
    fun setTargetingInfo(info: Map<String, List<String>>?) {
        logAATKitCall("CMD: setTargetingInfo($info)")
        adController!!.setTargetingInfo(null, info)
    }

    /**
     * Sets the targeting information for the given placement. Information provided for placement overrides targeting information for application set by the [.setTargetingInfo].
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param info        Map with targeting information
     */
    @JvmStatic
    fun setTargetingInfo(placementId: Int, info: Map<String, List<String>>) {
        logAATKitCall("CMD: setTargetingInfo($placementId, $info)")
        adController!!.setTargetingInfo(placementId, info)
    }

    /**
     * Sets the content targeting url for the application. This information will be used only if no placement-specific targeting is available.
     *
     * @param targetingUrl The targeting url
     * @see .setContentTargetingUrl
     */
    @JvmStatic
    fun setContentTargetingUrl(targetingUrl: String) {
        logAATKitCall("CMD: setContentTargetingUrl($targetingUrl)")
        adController!!.setContentTargetingUrl(null, targetingUrl)
    }

    /**
     * Sets the content targeting url for the given placement. Information provided for placement overrides targeting information for application set by the [.setContentTargetingUrl].
     *
     * @param placementId  Placement identifier obtained from [.createPlacement].
     * @param targetingUrl The targeting url
     */
    @JvmStatic
    fun setContentTargetingUrl(placementId: Int, targetingUrl: String?) {
        logAATKitCall("CMD: setContentTargetingUrl($placementId, $targetingUrl)")
        adController!!.setContentTargetingUrl(placementId, targetingUrl)
    }

    /**
     * Adds an ad network to the list of ad networks that receive targeting keywords (if any set).
     * If no ad networks are added, any set keywords will be delivered to all ad networks supporting keyword targeting.
     *
     * @param network Chosen ad network.
     */
    @JvmStatic
    fun addAdNetworkForKeywordTargeting(network: AdNetwork) {
        logAATKitCall("CMD: addAdNetworkForKeywordTargeting($network)")
        adController!!.addAdNetworkForKeywordTargeting(network)
    }

    /**
     * Removes an ad network from the list of ad networks that receive targeting keywords (if any set).
     * If no ad networks are added to the list, any set keywords will be delivered to all ad networks supporting keyword targeting.
     *
     * @param network Chosen ad network.
     */
    @JvmStatic
    fun removeAdNetworkForKeywordTargeting(network: AdNetwork) {
        logAATKitCall("CMD: removeAdNetworkForKeywordTargeting($network)")
        adController!!.removeAdNetworkForKeywordTargeting(network)
    }

    /**
     * Sets gravity for ads that don't fill entire placement area. Works only for banner placements.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param gravity     The [Gravity] to set.
     */
    @JvmStatic
    fun setPlacementContentGravity(placementId: Int, gravity: Int) {
        logAATKitCall("CMD: setPlacementContentGravity($placementId,$gravity)")
        adController!!.setPlacementContentGravity(placementId, gravity)
    }

    /**
     * Allows setting of ad rules that will be used before real rules from the server are downloaded.
     *
     * @param rules String containing the rules to be used.
     */
    @JvmStatic
    fun setInitialRules(rules: String) {
        logAATKitCall("CMD: setInitialRules($rules)")
        adController!!.setInitialRules(rules)
    }

    /**
     * Allows the AATKit to preserve last downloaded ad rules when the application is closed. Such rules will be re-used next time the application is started, before new ones get downloaded.
     *
     * @param enabled True to enable, false to disable.
     */
    @JvmStatic
    fun setRuleCachingEnabled(enabled: Boolean) {
        logAATKitCall("CMD: setRuleCachingEnabled($enabled)")
        adController!!.setRuleCachingEnabled(enabled)
    }

    /**
     * Allows to set log level from code.
     *
     * @param logLevel Desired log level, as in [android.util.Log] class.
     */
    @JvmStatic
    fun setLogLevel(logLevel: Int) {
        log("CMD: setLogLevel($logLevel)")
        adController!!.setLogLevel(logLevel)
    }

    /**
     * Allows to mute video ads.
     *
     * @param mute True to enable, false to disable ad muting.
     */
    @JvmStatic
    fun muteVideoAds(mute: Boolean) {
        logAATKitCall("CMD: muteVideoAds($mute)")
        adController!!.muteVideoAds(mute)
    }

    /**
     * Reports the VAST ad impression.
     *
     * @param data The data for a VAST ad which has been impressed.
     */
    @JvmStatic
    fun reportVASTImpression(data: VASTAdData) {
        logAATKitCall("CMD: reportVASTImpression($data)")
        adController!!.reportVASTImpression(data)
    }

    /**
     * Reports the VAST ad viewable impression.
     *
     * @param data The VAST ad data for which the viewable impression should be counted.
     */
    @JvmStatic
    fun reportVASTViewableImpression(data: VASTAdData) {
        logAATKitCall("CMD: reportVASTViewableImpression($data)")
        adController!!.reportVASTViewableImpression(data)
    }

    /**
     * Reports the VAST ad click.
     *
     * @param data The data for a VAST ad which has been clicked.
     */
    @JvmStatic
    fun reportVASTClick(data: VASTAdData) {
        logAATKitCall("CMD: reportVASTClick($data)")
        adController!!.reportVASTClick(data)
    }

    /**
     * Sets VAST ad request parameters.
     *
     * @param placementId Placement identifier obtained from [.createPlacement].
     * @param parameters  Parameters to set for a given placement identifier.
     */
    @JvmStatic
    fun setVASTRequestParameters(placementId: Int, parameters: VASTRequestParameters) {
        logAATKitCall("CMD: setVASTRequestParameters($placementId, $parameters)")
        adController!!.setVASTRequestParameters(placementId, parameters)
    }

    //region non-public methods
    private fun logAATKitCall(call: String) {
        if (ServerLogger.shouldLog(ServerLogger.Event.LOGCMD)) {
            log(call)
        }
        if (isLoggable(Log.VERBOSE)) {
            val command = call.replaceFirst("^CMD:\\s*".toRegex(), "")
            v(AATKit::class.java, command)
        }
    }

    internal fun createBannerPlacementForCache(
        placementName: String,
        configuration: BannerConfiguration?,
        cache: BannerCache,
        statsListener: StatisticsListener?
    ): Pair<BannerPlacement, Boolean>? {
        return adController!!.createBannerPlacementForCache(
            placementName,
            configuration,
            cache,
            statsListener
        )
    }

    internal fun destroyBannerCache(cache: BannerCache) {
        adController!!.destroyBannerCache(cache)
    }

    // methods below are intended only for use by the plugins.
    @JvmStatic
    fun destroy() {
        adController!!.destroy()
        adController = null
    }

    @JvmStatic
    val isInitialized: Boolean
        get() = adController != null

    /**
     * Shows AATKit debug screen.
     */
    @JvmStatic
    fun showDebugDialog() {
        logAATKitCall("CMD: showDebugDialog()")
        adController!!.showDebugDialog()
    }
    //endregion

    /**
     * Defines possible positions for AdChoices icon for Google native ads.
     */
    enum class AdChoicesIconPosition {
        TOP_LEFT, TOP_RIGHT, BOTTOM_LEFT, BOTTOM_RIGHT
    }

    /**
     * Used for passing of detailed GDPR consent. Possible implementations: [ManagedConsent], [SimpleConsent], [VendorConsent].
     */
    interface Consent {
        /**
         * Sets to set the networks that will not be requested if they do not have the TCF2 consent. Only works if TCF2 compliant CMP is used.
         *
         * @param stopSet Set of networks not to be requested without TCF2 consent.
         */
        fun setNoConsentNetworkStopSet(stopSet: Set<AdNetwork>?)
    }

    /**
     * Notifies about AATKit events.
     */
    interface Delegate {
        /**
         * Notifies that placement has finished loading an ad successfully.
         *
         * @param placementId Placement identifier obtained from [.createPlacement].
         */
        fun aatkitHaveAd(placementId: Int)

        /**
         * Notifies that placement has failed to load an ad.
         *
         * @param placementId Placement identifier obtained from [.createPlacement].
         */
        fun aatkitNoAd(placementId: Int)

        /**
         * Notifies that ad went fullscreen and that application should pause.
         *
         * @param placementId Placement identifier obtained from [.createPlacement].
         */
        fun aatkitPauseForAd(placementId: Int)

        /**
         * Notifies that ad came back from fullscreen and that application should resume.
         *
         * @param placementId Placement identifier obtained from [.createPlacement].
         */
        fun aatkitResumeAfterAd(placementId: Int)

        /**
         * Notifies that placement has loaded an Empty ad.
         *
         * @param placementId Placement identifier obtained from [.createPlacement].
         */
        fun aatkitShowingEmpty(placementId: Int)

        /**
         * Notifies that placement has earned incentive (by rewarded ads).
         *
         *
         * Note: This callback might also be invoked for regular fullscreen
         * placements, not just the ones configured for rewarded videos. Please
         * make sure placementId is your rewarded video placement before rewarding
         * the user.
         *
         * @param placementId  Placement identifier obtained from [.createPlacement].
         * @param aatKitReward Incentive object. Can be null for networks not supporting reward information.
         */
        fun aatkitUserEarnedIncentive(placementId: Int, aatKitReward: AATKitReward?)

        /**
         * Notifies that the AATKit has obtained ad rules.
         *
         * @param fromTheServer Indicates if the rules came from the server. It will return false if the currently used rules come from the [.setInitialRules] method or the cached rules are used.
         */
        fun aatkitObtainedAdRules(fromTheServer: Boolean)

        /**
         * Notifies that application's bundle ID was not recognized by the AddApptr server.
         */
        fun aatkitUnknownBundleId()

        /**
         * Notifies that AATKit has new banner view ready for placement. Used only for MultiSizeBanner.
         *
         * @param placementId Placement identifier obtained from [.createPlacement].
         * @param bannerView  Loaded banner view
         */
        fun aatkitHaveAdForPlacementWithBannerView(
            placementId: Int,
            bannerView: BannerPlacementLayout
        )

        /**
         * Notifies that placement has finished loading an VAST ad successfully. Used only for VAST.
         *
         * @param placementId Placement identifier obtained from [.createPlacement].
         * @param data        Loaded VAST ad data.
         */
        fun aatkitHaveVASTAd(placementId: Int, data: VASTAdData)
    }

    /**
     * Notifies about placement reporting events, like counted adspace, request etc.
     */
    interface StatisticsListener {
        /**
         * Notifies that an adspace has been counted.
         */
        fun countedAdSpace()

        /**
         * Notifies that an request has been counted for a given network.
         *
         * @param network Network for which the request has been counted.
         */
        fun countedRequest(network: AdNetwork)

        /**
         * Notifies that an response has been counted for a given network.
         *
         * @param network Network for which the response has been counted.
         */
        fun countedResponse(network: AdNetwork)

        /**
         * Notifies that an impression has been counted for a given network.
         *
         * @param network Network for which the impression has been counted.
         */
        fun countedImpression(network: AdNetwork)

        /**
         * Notifies that an viewable impression has been counted for a given network.
         *
         * @param network Network for which the viewable impression has been counted.
         */
        fun countedVimpression(network: AdNetwork)

        /**
         * Notifies that an direct deal impression has been counted for a given network.
         *
         * @param network Network for which the direct deal impression has been counted.
         */
        fun countedDirectDealImpression(network: AdNetwork)

        /**
         * Notifies that an click has been counted for a given network.
         *
         * @param network Network for which the click has been counted.
         */
        fun countedClick(network: AdNetwork)
    }
}