package com.intentsoftware.addapptr

import android.app.Application
import android.content.pm.PackageManager
import android.os.Handler
import android.util.Log
import com.intentsoftware.addapptr.AATKit.StatisticsListener
import com.intentsoftware.addapptr.AATKit.createBannerPlacementForCache
import com.intentsoftware.addapptr.AATKit.destroyBannerCache
import com.intentsoftware.addapptr.internal.Placement
import com.intentsoftware.addapptr.internal.ad.Ad.ExpirationListener
import com.intentsoftware.addapptr.internal.module.Logger.d
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.isLoggable
import com.intentsoftware.addapptr.internal.module.Logger.v
import com.intentsoftware.addapptr.internal.module.Logger.w
import java.util.*

/**
 * A cache of automatically preloaded banner ads. The cache will always try to have a defined amount of banners available for immediate handout to the app whenever they are needed.
 * **The BannerCache needs to be destroyed when no longer needed.**
 */
@Suppress("unused")
class BannerCache @JvmOverloads constructor(
    cacheConfiguration: BannerCacheConfiguration,
    statisticsListener: StatisticsListener? = null
) {
    private val configuration: BannerCacheConfiguration = BannerCacheConfiguration(cacheConfiguration)
    private var bannerPlacement: BannerPlacement?
    private val handler = Handler()
    private val loadedBanners: Queue<BannerPlacementLayout> = LinkedList()
    private val runningRequests: MutableList<BannerRequest> = ArrayList()
    private val bannerReloadRunnables: MutableList<Runnable> =
        ArrayList() //list for runnables for retrying failed banner requests
    private var consumptionTimestamp: Long = 0
    private var destroyed = false
    private var shouldNotifyDelegate = true
    private var noFills //number of failed requests, resets on successful load
            = 0

    //region Public API

    /**
     * Sets the impression listener for banner cache.
     *
     * @param impressionListener [ImpressionListener] implementation that will be notified about impression events.
     */
    fun setImpressionListener(impressionListener: ImpressionListener) {
        if (checkDestroyedOrFailedToInitialize("updateRequestConfiguration")) return
        if (isLoggable(Log.VERBOSE)) {
            v(this, "Setting impression listener " + impressionListener + " for cache " + toString())
        }
        if (bannerPlacement is Placement) {
            val placement = bannerPlacement as Placement
            placement.setImpressionListener(impressionListener)
        } else {
            if (isLoggable(Log.WARN)) {
                w(
                    this,
                    "Set impression listener for cache " + toString() + "failed, banner placement is not an instance of supported classes!"
                )
            }
        }
    }

    /**
     * Updates the configuration that will be used when requesting new banners.
     *
     * @param requestConfiguration [BannerRequest] instance. Can not be null.
     * @param shouldRefresh        True if the whole cache should be re-loaded with new banner request configuration, false if new configuration should only be used for new requests.
     */
    fun updateRequestConfiguration(requestConfiguration: BannerRequest?, shouldRefresh: Boolean) {
        if (checkDestroyedOrFailedToInitialize("updateRequestConfiguration")) return
        if (isLoggable(Log.VERBOSE)) {
            v(
                this,
                "Updating request configuration with configuration:" + requestConfiguration + ", shouldRefresh:" + shouldRefresh + " for cache:" + toString()
            )
        }
        configuration.requestConfiguration = requestConfiguration
        if (shouldRefresh) {
            shouldNotifyDelegate = true //we want to inform the delegate again
            clearCache()
            cancelRunningRequests()
            fillCache()
        }
    }

    /**
     * Returns an instance of [BannerPlacementLayout] to be used within the app. Also automatically counts an ad space.
     *
     *
     * **BannerCache will no longer hold any references to returned banners, and they need to be destroyed manually by the app.**
     *
     *
     * This method respects the frequency capping, set by [BannerCacheConfiguration.setMinimumDelay]
     *
     * @return [BannerPlacementLayout] instance.
     */
    @Synchronized
    fun consume(): BannerPlacementLayout? {
        return consume(false)
    }

    /**
     * Returns an instance of [BannerPlacementLayout] to be used within the app. Also automatically counts an ad space.
     * **BannerCache will no longer hold any references to returned banners, and they need to be destroyed manually by the app.**
     *
     * @param force true if cache should try to return banner ignoring the frequency capping set by [BannerCacheConfiguration.setMinimumDelay]
     * @return [BannerPlacementLayout] instance.
     */
    @Synchronized
    fun consume(force: Boolean): BannerPlacementLayout? {
        if (checkDestroyedOrFailedToInitialize("consume")) return null
        if (isLoggable(Log.VERBOSE)) {
            v(this, "Called consume method with param force=$force")
        }
        val currentTime = System.currentTimeMillis()
        return if (currentTime - consumptionTimestamp >= configuration.minDelayMillis || force) {
            bannerPlacement?.countAdSpace()
            val banner = loadedBanners.poll()
            if (isLoggable(Log.VERBOSE)) {
                v(
                    this,
                    "Consuming banner from cache:" + toString() + ", returning:" + banner
                )
            }
            fillCache()
            if (banner != null) {
                banner.setExpirationListener(null) //clear the expiration listener, we do not want to touch already returned ads
                consumptionTimestamp = currentTime
            }
            banner
        } else {
            if (isLoggable(Log.VERBOSE)) {
                v(
                    this,
                    "Minimum delay between \"consume\" calls not reached, returning null"
                )
            }
            null
        }
    }

    /**
     * Destroys the BannerCache, clearing all preloaded banner ads and canceling pending reload requests. For proper memory management, it needs to be called when the BannerCache is no longer needed.
     * **Destroyed BannerCache can no longer be used.**
     */
    fun destroy() {
        if (checkDestroyedOrFailedToInitialize("destroy")) return
        if (isLoggable(Log.VERBOSE)) {
            v(this, "Destroying cache:" + toString())
        }
        destroyBannerCache(this)
        cancelRunningRequests()
        clearCache()
        configuration.delegate = null
        configuration.requestConfiguration = null
        destroyed = true
    }

    //endregion
    //region lifecycle handling
    internal fun onResume() {
        noFills = 0 //reset the counter
        fillCache()
    }

    internal fun onPause() {
        cancelRunningRequests()
    }

    //endregion
    //region private methods
    private fun checkDestroyedOrFailedToInitialize(methodName: String): Boolean {
        if (bannerPlacement == null) {
            if (isLoggable(Log.ERROR)) {
                e(this, "Cannot execute method $methodName, BannerCache was not created properly.")
            }
            return true
        } else if (destroyed) {
            if (isLoggable(Log.WARN)) {
                w(this, "Cannot execute method $methodName, BannerCache is already destroyed!")
            }
        }
        return destroyed
    }

    @Synchronized
    private fun fillCache() {
        if ((bannerPlacement as Placement?)?.isActivityResumed == false) {
            if (isLoggable(Log.DEBUG)) {
                d(this, "Cannot fill cache, activity is paused")
            }
            return
        }
        var adsToLoad = configuration.size - loadedBanners.size - runningRequests.size - bannerReloadRunnables.size
        if (adsToLoad > 0) {
            if (configuration.shouldCacheAdditionalAdAtStart()) {
                adsToLoad++
            }
            if (isLoggable(Log.VERBOSE)) {
                v(this, "Banner cache " + toString() + " will request:" + adsToLoad + " banners.")
            }
            configuration.setShouldCacheAdditionalAdAtStart(false) //AFTER log, so that toString() won't lie about extra ad

            for (i in 0 until adsToLoad) {
                requestAd()
            }
        }
    }

    @Synchronized
    private fun clearCache() {
        for (banner in loadedBanners) {
            banner.destroy()
        }
        loadedBanners.clear()
    }

    @Synchronized
    private fun cancelRunningRequests() {
        if (isLoggable(Log.VERBOSE)) {
            v(this, "Canceling running requests")
        }
        val requestsToBeCanceled: List<BannerRequest> =
            ArrayList(runningRequests) //needs a copy, as request.cancel ends up in modifying the list and thus ConcurrentModificationException
        runningRequests.clear()
        for (request in requestsToBeCanceled) {
            bannerPlacement?.cancel(request)
        }
        for (runnable in bannerReloadRunnables) {
            handler.removeCallbacks(runnable)
        }
        bannerReloadRunnables.clear()
    }

    @Synchronized
    private fun handleAdExpired(layout: BannerPlacementLayout) {
        if (destroyed) {
            if (isLoggable(Log.VERBOSE)) {
                v(this, "Ad expiration ignored, banner cache has been destroyed")
            }
            return
        }
        if (isLoggable(Log.VERBOSE)) {
            v(this, "Ad has expired ad will be removed from cache")
        }
        loadedBanners.remove(layout)
        layout.destroy()
        if ((bannerPlacement as Placement?)?.isActivityResumed == true) {
            requestAd()
        }
    }

    @Synchronized
    private fun requestAd() {
        if ((bannerPlacement as Placement?)?.isActivityResumed == false) {
            if (isLoggable(Log.DEBUG)) {
                d(this, "Cannot request ad, activity is paused")
            }
            return
        }
        if (destroyed) {
            if (isLoggable(Log.DEBUG)) {
                d(this, "Cannot request ad, cache is destroyed")
            }
            return
        }
        if (noFills >= configuration.size && runningRequests.size + bannerReloadRunnables.size >= 1) {
            if (isLoggable(Log.VERBOSE)) {
                v(
                    this,
                    "Ad will not be requested due to number of failed requests:" + noFills + " and reload request already being posted " +
                            "(Running requests: " + runningRequests.size + ", reload runnables posted: " + bannerReloadRunnables.size + ")."
                )
            }
            return
        }
        val newRequest = BannerRequest(configuration.requestConfiguration?.delegate)
        newRequest.setBannerSizes(configuration.requestConfiguration?.getBannerSizes())
        newRequest.targetingInformation = configuration.requestConfiguration?.targetingInformation
        newRequest.contentTargetingUrl = configuration.requestConfiguration?.contentTargetingUrl
        runningRequests.add(newRequest)
        bannerPlacement?.requestAd(newRequest, createRequestCompletionListener(newRequest))
    }

    private fun createRequestCompletionListener(request: BannerRequest): BannerRequestCompletionListener {
        return object : BannerRequestCompletionListener {
            override fun onRequestCompleted(layout: BannerPlacementLayout?, error: BannerRequestError?) {
                synchronized(this@BannerCache) {
                    runningRequests.remove(request)
                    if (layout != null) {
                        noFills = 0 //reset the counter
                        if (isLoggable(Log.VERBOSE)) {
                            v(this@BannerCache, "Banner:" + layout + " added to cache:" + this@BannerCache.toString())
                        }
                        layout.setExpirationListener(object : ExpirationListener {
                            override fun onAdExpired() {
                                handleAdExpired(layout)
                            }
                        })

                        loadedBanners.add(layout)
                        if (shouldNotifyDelegate && configuration.delegate != null) {
                            if (isLoggable(Log.VERBOSE)) {
                                v(
                                    this@BannerCache,
                                    "Banner cache:" + this@BannerCache.toString() + " notifying delegate:" + configuration.delegate
                                )
                            }
                            configuration.delegate?.firstBannerLoaded()
                        }
                        shouldNotifyDelegate = false
                        fillCache()
                    } else if (!request.isCancelled) { //do not treat canceled requests as failure
                        noFills++
                        if (isLoggable(Log.VERBOSE)) {
                            v(this@BannerCache, "Failed to load banner for queue, error:" + error?.message)
                        }
                        val runnable: Runnable = object : Runnable {
                            override fun run() {
                                synchronized(this@BannerCache) {
                                    bannerReloadRunnables.remove(this)
                                    requestAd()
                                }
                            }
                        }
                        bannerReloadRunnables.add(runnable)
                        handler.postDelayed(
                            runnable,
                            onFailedBannerReloadInterval.toLong()
                        ) //retry banner reload after given time
                    }
                }
            }
        }
    }

    //endregion
    override fun toString(): String {
        return "BannerCache{" +
                "configuration=" + configuration +
                '}'
    }

    /**
     * Optional delegate informing about events in BannerCache.
     */
    interface CacheDelegate {
        /**
         * Called when the first banner gets loaded for the cache. Only called once.
         */
        fun firstBannerLoaded()
    }

    companion object {
        private const val BANNER_CACHE_RELOAD_INTERVAL_METADATA_TAG =
            "com.intentsoftware.addapptr.banner_cache_reload_interval"
        private const val ON_FAILED_BANNER_RELOAD_INTERVAL_DEFAULT = 10000
        private var onFailedBannerReloadInterval = ON_FAILED_BANNER_RELOAD_INTERVAL_DEFAULT

        //called by AdController, as we need an Context instance
        @JvmStatic
        internal fun init(application: Application) {
            try {
                val ai =
                    application.packageManager.getApplicationInfo(application.packageName, PackageManager.GET_META_DATA)
                val bundle = ai.metaData
                val reloadInterval = bundle.getInt(BANNER_CACHE_RELOAD_INTERVAL_METADATA_TAG)
                if (reloadInterval > 0) {
                    if (isLoggable(Log.VERBOSE)) {
                        v(
                            BannerCache::class.java,
                            "Found value:$reloadInterval for BannerCache banner reload interval."
                        )
                    }
                    onFailedBannerReloadInterval = reloadInterval * 1000
                } else {
                    if (isLoggable(Log.VERBOSE)) {
                        v(BannerCache::class.java, "No value for BannerCache banner reload interval found.")
                    }
                }
            } catch (e: Exception) {
                if (isLoggable(Log.WARN)) {
                    w(
                        BannerCache::class.java,
                        "Exception when looking for BannerCache banner reload interval in manifest",
                        e
                    )
                }
            }
        }
    }
    /**
     * Creates a cache of automatically preloaded banner ads.
     *
     * @param cacheConfiguration [BannerCacheConfiguration] instance used to configure the cache.
     * @param statisticsListener [AATKit.StatisticsListener] implementation that will be notified about statistics events.
     */

    /**
     * Creates a cache of automatically preloaded banner ads.
     *
     * @param cacheConfiguration [BannerCacheConfiguration] instance used to configure the cache.
     */
    init {
        val configuration = BannerConfiguration()
        configuration.isManualAdSpaceCounting = true
        val placementBooleanPair =
            createBannerPlacementForCache(cacheConfiguration.placementName, configuration, this, statisticsListener)

        if (placementBooleanPair == null) {
            if (isLoggable(Log.ERROR)) {
                e(
                    this,
                    "Failed to create banner placement with name:" + cacheConfiguration.placementName + " for banner cache:" + toString() + ", cache will not work."
                )
            }
            bannerPlacement = null
        } else {
            bannerPlacement = placementBooleanPair.first
            if (bannerPlacement == null) {
                if (isLoggable(Log.ERROR)) {
                    e(
                        this,
                        "Failed to create banner placement with name:" + cacheConfiguration.placementName + " for banner cache:" + toString() + ", cache will not work."
                    )
                }
            } else if (isLoggable(Log.VERBOSE)) {
                v(
                    this,
                    "Created banner cache:" + toString() + " with placement name:" + cacheConfiguration.placementName + " statistics listener:" + statisticsListener
                )
            }
            if (placementBooleanPair.second) {
                onResume()
            }
        }
    }
}