package com.intentsoftware.addapptr

import android.app.Activity
import android.content.Context
import android.util.Log
import com.intentsoftware.addapptr.consent.CMP
import com.intentsoftware.addapptr.consent.CMPDelegate
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.intentsoftware.addapptr.internal.ConsentHelper
import com.intentsoftware.addapptr.internal.ConsentImplementation
import com.intentsoftware.addapptr.internal.module.Logger
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.isLoggable
import com.intentsoftware.addapptr.internal.module.Logger.v
import com.intentsoftware.addapptr.internal.module.Logger.w

/**
 * Will present a dialog for the users to grant or withhold their consent.
 */
class ManagedConsent(cmp: CMP, context: Context, delegate: ManagedConsentDelegate) : ConsentImplementation(),
    CMPDelegate {
    private val cmp: CMPImplementation?
    private val delegate: ManagedConsentDelegate?
    private val applicationContext: Context?

    /**
     * Presents the consent screen ONLY if it is required by the used CMP (i.e. no user consent has been set yet).
     *
     * @param activity The Activity from which the consent screen will be shown.
     */
    fun showIfNeeded(activity: Activity) {
        if (isLoggable(Log.VERBOSE)) {
            v(this, "showIfNeeded called")
        }
        if (cmp != null && cmp.isSuccessfullyInitialized) {
            cmp.showIfNeeded(activity)
        } else {
            if (isLoggable(Log.ERROR)) {
                e(this, "ManagedConsent was not initialized with a working CMP. Cannot show.")
            }
        }
    }

    /**
     * Presents the consent screen, allowing the user to change consent settings.
     *
     * @param activity The Activity from which the consent screen will be shown.
     */
    fun editConsent(activity: Activity) {
        if (isLoggable(Log.VERBOSE)) {
            v(this, "editConsent called")
        }
        if (cmp != null && cmp.isSuccessfullyInitialized) {
            cmp.editConsent(activity)
        } else {
            if (isLoggable(Log.ERROR)) {
                e(this, "ManagedConsent was not initialized with a working CMP. Cannot edit.")
            }
        }
    }

    /**
     * Tells the CMP to reload. You can call this method for example after receiving [ManagedConsentDelegate.managedConsentCMPFailedToLoad] callback.
     *
     * @param activity The Activity instance.
     */
    fun reload(activity: Activity) {
        if (isLoggable(Log.VERBOSE)) {
            v(this, "reload called")
        }
        if (cmp != null && cmp.isSuccessfullyInitialized) {
            cmp.reload(activity)
        } else {
            if (isLoggable(Log.ERROR)) {
                e(this, "ManagedConsent was not initialized with a working CMP. Cannot reload.")
            }
        }
    }

    override fun onConsentUpdated(state: ManagedConsentState) {
        if (applicationContext != null) {
            readConsentStringFromSharedPreferences(applicationContext)
            ConsentHelper.reconfigureNetworks(applicationContext)
        } else {
            if (Logger.isLoggable(Log.ERROR)) {
            	Logger.e(this, "Cannot handle consent update, application context is null")
            }
        }
        delegate?.managedConsentCMPFinished(state)
    }

    override fun onCMPFailedToShow(error: String) {
        if (isLoggable(Log.WARN)) {
            w(this, "Failed to show CMP: $error")
        }
        delegate?.managedConsentCMPFailedToShow(this, error)
    }

    override fun onCMPFailedToLoad(error: String) {
        if (isLoggable(Log.WARN)) {
            w(this, "Failed to load CMP: $error")
        }
        delegate?.managedConsentCMPFailedToLoad(this, error)
    }

    override fun CMPNeedsUI() {
        if (isLoggable(Log.VERBOSE)) {
            v(this, "CMP needs UI")
        }
        delegate?.managedConsentNeedsUserInterface(this)
    }

    override fun getConsentForNetwork(network: AdNetwork): NonIABConsent {
        return if (cmp != null && cmp.isSuccessfullyInitialized) {
            cmp.getConsentForNetwork(network)
        } else {
            if (isLoggable(Log.ERROR)) {
                e(
                    this,
                    "ManagedConsent was not initialized with a working CMP, cannot check consent for network"
                )
            }
            NonIABConsent.WITHHELD
        }
    }

    override fun toString(): String {
        return "ManagedConsent{" +
                "cmp=" + cmp +
                ", delegate=" + delegate +
                "} " + super.toString()
    }

    /**
     * Notifies about the need to show the consent dialog.
     */
    interface ManagedConsentDelegate {
        /**
         * Notifies that [ManagedConsent] needs to show a consent dialog. After receiving this notification, you should pause your application and call the [.showIfNeeded] method.
         *
         * @param managedConsent The [ManagedConsent] instance that requests UI.
         */
        fun managedConsentNeedsUserInterface(managedConsent: ManagedConsent)

        /**
         * Notifies that the used [CMP] has finished updating consent.
         * @param state The [ManagedConsentState] informing about consent given by the user.
         */
        fun managedConsentCMPFinished(state: ManagedConsentState)

        /**
         * Notifies that the used [CMP] failed to load.
         *
         * @param managedConsent @param managedConsent The [ManagedConsent] instance.
         * @param error          The description of what went wrong
         */
        fun managedConsentCMPFailedToLoad(managedConsent: ManagedConsent, error: String?)

        /**
         * Notifies that the used [CMP] failed to show.
         *
         * @param managedConsent @param managedConsent The [ManagedConsent] instance.
         * @param error          The description of what went wrong
         */
        fun managedConsentCMPFailedToShow(managedConsent: ManagedConsent, error: String?)
    }

    /**
     * Possible states of consent given by the user.
     */
    enum class ManagedConsentState {
        /**
         * No information about consent state.
         */
        UNKNOWN,

        /**
         * Consent has been declined by the user.
         */
        WITHHELD,

        /**
         * Partial consent has been granted by the user - at least some purposes and some vendors were given consent.
         */
        CUSTOM,

        /**
         * Full consent has been granted by the user.
         */
        OBTAINED
    }

    /**
     * Creates the [ManagedConsent] instance.
     *
     * @param cmp      Instance of [CMP] to be used.
     * @param context  The [Context] of your application.
     * @param delegate The [ManagedConsentDelegate] instance. Must not be null.
     */
    init {
        if (cmp is CMPImplementation) {
            if (cmp.isSuccessfullyInitialized) {
                this.cmp = cmp
                this.cmp.setDelegate(this)
                this.delegate = delegate
                this.applicationContext = context.applicationContext
                if (isLoggable(Log.VERBOSE)) {
                    v(this, "initialized ManagedConsent: " + toString())
                }
            } else {
                if (isLoggable(Log.ERROR)) {
                    e(this, "Passed CMP was not successfully initialized. ManagedConsent will not work.")
                }
                this.cmp = null
                this.delegate = null
                this.applicationContext = null
            }
        } else {
            if (isLoggable(Log.ERROR)) {
                e(this, "Passed CMP is not an instance of allowed classes. ManagedConsent will not work.")
            }
            this.cmp = null
            this.delegate = null
            this.applicationContext = null
        }
    }
}