@file:Suppress("unused")

package com.intentsoftware.addapptr

import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.util.Log
import android.util.Pair
import com.intentsoftware.addapptr.internal.AdController
import com.intentsoftware.addapptr.internal.BannerCacheImplementation
import com.intentsoftware.addapptr.internal.BannerReloader
import com.intentsoftware.addapptr.internal.InfeedBannerPlacementImplementation
import com.intentsoftware.addapptr.internal.InfeedConfigsHandler
import com.intentsoftware.addapptr.internal.InfeedPlacementData
import com.intentsoftware.addapptr.internal.SupportedNetworks
import com.intentsoftware.addapptr.internal.Version
import com.intentsoftware.addapptr.internal.module.Logger
import com.intentsoftware.addapptr.internal.module.Logger.d
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.i
import com.intentsoftware.addapptr.internal.module.Logger.w
import com.intentsoftware.addapptr.internal.module.ServerLogger
import com.intentsoftware.addapptr.internal.module.ServerLogger.log
import com.intentsoftware.addapptr.internal.module.Utils
import com.intentsoftware.addapptr.internal.module.Utils.fittingBannerSizes
import com.intentsoftware.addapptr.internal.module.Utils.maxPlacementSize
import com.intentsoftware.addapptr.internal.module.Utils.screenWidthLandscape
import com.intentsoftware.addapptr.internal.module.Utils.screenWidthPortrait
import com.intentsoftware.addapptr.internal.module.debugscreen.AATKitDebugInfo

object AATKit {

    private const val BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS =
        (BannerReloader.AUTORELOAD_INTERVAL_MIN / 1000).toInt() //for now KDoc does not support {@value} tags...
    private const val BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS =
        (BannerReloader.AUTORELOAD_INTERVAL_MAX / 1000).toInt()

    @JvmSynthetic
    internal var adController: AdController? = null

    /**
     * @return AATKit version.
     */
    @JvmStatic
    val version: String
        get() {
            logAATKitCall("CMD: getVersion()")
            return Version.NAME
        }

    /**
     * @return AATKit detailed build version.
     */
    @JvmStatic
    val fullVersion: String
        get() {
            logAATKitCall("CMD: getFullVersion()")
            return Version.FULL_NAME
        }

    /**
     * Initializes the AATKit library. Should be called once during application initialization before any other calls to AATKit.
     *
     * @param configuration Configuration for AATKit.
     * @see AATKitConfiguration
     */
    @JvmStatic
    fun init(configuration: AATKitConfiguration) {
        logAATKitCall("CMD: init($configuration)", true)
        if (adController != null) {
            e(AATKit::class.java, { "AdController is already initialized" })
            return
        }
        var version = "?"
        val packageName = configuration.application.applicationContext.packageName
        try {
            version =
                configuration.application.packageManager.getPackageInfo(packageName, 0).versionName ?: "?"
        } catch (e: PackageManager.NameNotFoundException) {
            w(AATKit::class.java, { "Failed to check version of application" }, e)
        }
        d("Init", { (Version.FULL_NAME + ", application: " + packageName + " (version: " + version + ")" + ", configuration: " + configuration) }, true)
        adController = AdController(configuration)
    }

    /**
     * Enables debug screen that will show after shaking the device. It is already enabled by default.
     */
    @JvmStatic
    fun enableDebugScreen() {
        logAATKitCall("CMD: enableDebugScreen()")
        adController?.enableDebugScreen() ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Disables the debug screen appearing after shaking the device. It is enabled by default.
     */
    @JvmStatic
    fun disableDebugScreen() {
        logAATKitCall("CMD: disableDebugScreen()")
        adController?.disableDebugScreen() ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Used for obtaining debug information (the same that would be presented in dialog after shaking the device if debug screen is enabled)
     *
     * @return String with debug information
     */
    @JvmStatic
    val debugInfo: String
        get() {
            logAATKitCall("CMD: getDebugInfo()")
            return adController?.debugInfo ?: "AATKit not initialized"
        }

    /**
     * Allows to reconfigure the options for GDPR consent.
     *
     * @param configuration New configuration.
     */
    @JvmStatic
    fun reconfigure(configuration: AATKitRuntimeConfiguration) {
        logAATKitCall("CMD: reconfigure($configuration)")
        adController?.reconfigure(configuration) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Checks if AATKit recognizes given device as tablet.
     *
     * @param context The [Context] of your application.
     * @return True if device is recognized as tablet, false otherwise.
     */
    @JvmStatic
    fun isTablet(context: Context): Boolean {
        logAATKitCall("CMD: isTablet($context)")
        return Utils.isTablet(context)
    }

    /**
     * Returns the [BannerPlacementSize] with maximum width that will fit on a given device in portrait screen orientation.
     *
     * @param context The [Context] of your application.
     * @return [BannerPlacementSize] best fitting current device
     */
    @JvmStatic
    fun maximumBannerSizePortrait(context: Context): BannerPlacementSize {
        logAATKitCall("CMD: maximumBannerSizePortrait($context)")
        return maxPlacementSize(screenWidthPortrait(context))
    }

    /**
     * Returns the [BannerPlacementSize] with maximum width that will fit on a given device in landscape screen orientation.
     *
     * @param context The [Context] of your application.
     * @return [BannerPlacementSize] best fitting current device
     */
    @JvmStatic
    fun maximumBannerSizeLandscape(context: Context): BannerPlacementSize {
        logAATKitCall("CMD: maximumBannerSizeLandscape($context)")
        return maxPlacementSize(screenWidthLandscape(context))
    }

    /**
     * Returns the set of [BannerSize] that will fit on a given device in portrait screen orientation.
     *
     * @param context The [Context] of your application.
     * @return Set of [BannerSize] fitting current device
     */
    @JvmStatic
    fun fittingBannerSizesPortrait(context: Context): Set<BannerSize> {
        logAATKitCall("CMD: fittingBannerSizesPortrait($context)")
        return fittingBannerSizes(screenWidthPortrait(context))
    }

    /**
     * Returns the set of [BannerSize] that will fit on a given device in landscape screen orientation.
     *
     * @param context The [Context] of your application.
     * @return Set of [BannerSize] fitting current device
     */
    @JvmStatic
    fun fittingBannerSizesLandscape(context: Context): Set<BannerSize> {
        logAATKitCall("CMD: fittingBannerSizesLandscape($context)")
        return fittingBannerSizes(screenWidthLandscape(context))
    }

    /**
     * Notifies AATKit about activity resume. Invoke this method in every activity that uses AATKit.
     *
     * @param activity Reference to [Activity].
     */
    @JvmStatic
    fun onActivityResume(activity: Activity) {
        logAATKitCall("CMD: onActivityResume($activity)")
        adController?.onActivityResume(activity) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Notifies AATKit about activity pause. Invoke this method in every activity that uses AATKit.
     *
     * @param activity Reference to [Activity].
     */
    @JvmStatic
    fun onActivityPause(activity: Activity) {
        logAATKitCall("CMD: onActivityPause($activity)")
        adController?.onActivityPause(activity) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Creates a new fullscreen placement. If the fullscreen placement of given name already exists, it will be returned.
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @return FullscreenPlacement instance, or null if placement cannot be created.
     */
    @JvmStatic
    fun createFullscreenPlacement(name: String): FullscreenPlacement? {
        logAATKitCall("CMD: createFullscreenPlacement($name)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createFullscreenPlacement(name)
    }

    /**
     * Creates a new  infeed banner placement. If the banner placement of given name already exists, it will be returned.
     *
     *
     * **The placement will create a copy of the
     * configuration.** Any changes made to the configuration after placement is created will be ignored.
     *
     * @param name          Unique name of placement. The same name will be used in addapptr.com account.
     * @param configuration The configuration for this placement. The placement will ignore any changes made to configuration after it was created.
     * @return Infeed banner placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createInfeedBannerPlacement(
        name: String,
        configuration: BannerConfiguration?
    ): InfeedBannerPlacement? {
        logAATKitCall("CMD: createInFeedBannerPlacement($name,$configuration)")
        val infeedPlacementData = InfeedPlacementData(
            configHandler = InfeedConfigsHandler(),
            size = BannerSize.MultipleSizes,
            useGlobalTargeting = false,
        )
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }

        return adController.createInfeedBannerPlacement(name, configuration, infeedPlacementData)
    }

    /**
     * Creates a new sticky banner placement. If the banner placement of given name already exists, it will be returned.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @param bannerPlacementSize Size of the sticky banner placement.
     * @return Sticky banner placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createStickyBannerPlacement(
        name: String,
        bannerPlacementSize: BannerPlacementSize
    ): StickyBannerPlacement? {
        logAATKitCall("CMD: createStickyBannerPlacement($name,$bannerPlacementSize)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createStickyBannerPlacement(name, bannerPlacementSize)
    }

    /**
     * Creates a new multisize banner placement. If the banner placement of given name already exists, it will be returned.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @return Multisize banner placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createMultiSizeBannerPlacement(name: String): MultiSizeBannerPlacement? {
        logAATKitCall("CMD: createMultiSizeBannerPlacement($name)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createMultiSizeBannerPlacemement(name)
    }

    /**
     * Creates a new automatically loading banner placement. If the banner placement of given name and size already exists, it will be returned.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @param bannerPlacementSize Size of the sticky banner placement.
     * @return AutoLoadBannerPlacement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createAutoLoadBannerPlacement(
        name: String,
        bannerPlacementSize: BannerPlacementSize
    ): AutoLoadBannerPlacement? {
        logAATKitCall("CMD: createAutoLoadBannerPlacement($name,$bannerPlacementSize)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createAutoLoadBannerPlacement(name, bannerPlacementSize)
    }

    /**
     * Creates a new automatically loading multisize banner placement. If the banner placement of given name already exists, it will be returned.
     *
     * @param name               Unique name of placement. The same name will be used in addapptr.com account.
     * @return AutoLoadMultiSizeBannerPlacement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createAutoLoadMultiSizeBannerPlacement(name: String): AutoLoadMultiSizeBannerPlacement? {
        logAATKitCall("CMD: createAutoLoadMultiSizeBannerPlacement($name)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createAutoLoadMultiSizeBannerPlacement(name)
    }

    /**
     * Creates a new rewarded video placement. If the rewarded video ad placement of given name already exists, it will be returned.
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @return Rewarded video placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createRewardedVideoPlacement(name: String): RewardedVideoPlacement? {
        logAATKitCall("CMD: createRewardedVideoPlacement($name)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createRewardedVideoPlacement(name)
    }

    /**
     * Creates a new native ad placement. If the native ad placement of given name already exists, it will be returned.
     *
     * @param name              Unique name of placement. The same name will be used in addapptr.com account.
     * @param supportsMainImage True if the native ads returned should have main image asset. Keep in mind that if main image is used, it has to be displayed.
     * @return NativeAd placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createNativeAdPlacement(name: String, supportsMainImage: Boolean): NativeAdPlacement? {
        logAATKitCall("CMD: createNativeAdPlacement($name,$supportsMainImage)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createNativeAdPlacement(name, supportsMainImage)
    }

    /**
     * Creates a new App Open Ad placement. If the App Open Ad placement of given name already exists, it will be returned.
     *
     * @param name Unique name of placement. The same name will be used in addapptr.com account.
     * @return App Open Ad placement instance, or null if the placement cannot be created.
     */
    @JvmStatic
    fun createAppOpenAdPlacement(name: String): AppOpenAdPlacement? {
        logAATKitCall("CMD: createAppOpenAdPlacement($name)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createAppOpenAdPlacement(name)
    }

    /**
     * Creates a cache of automatically preloaded banner ads. The cache will always try to have a defined amount of banners available for immediate handout to the app whenever they are needed.
     * **The BannerCache needs to be destroyed when no longer needed.**
     *
     * @param configuration [BannerCacheConfiguration] instance used to configure the cache.
     */
    @JvmStatic
    fun createBannerCache(configuration: BannerCacheConfiguration): BannerCache? {
        logAATKitCall("CMD: createBannerCache($configuration)")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.createBannerCache(configuration)
    }

    /**
     * Allows to set the desired position of AdChoices icon in Google native ads. Set null to use the ad network's default.
     *
     * @param position desired AdChoices icon position. Null by default.
     */
    @JvmStatic
    fun setAdChoicesIconPosition(position: AdChoicesIconPosition?) {
        logAATKitCall("CMD: setAdChoicesIconPosition($position)")
        adController?.setAdChoicesIconPosition(position) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Allows to enable or disable selected ad networks. By default all networks are enabled.
     *
     * @param network Ad network.
     * @param enabled True to enable, false to disable.
     */
    @JvmStatic
    fun setNetworkEnabled(network: AdNetwork, enabled: Boolean) {
        logAATKitCall("CMD: setNetworkEnabled($network,$enabled)")
        SupportedNetworks.setNetworkEnabled(network, enabled)
    }

    /**
     * Returns true if ad network is enabled, false otherwise.
     *
     * @param network Ad network.
     * @return True if ad network is enabled, false otherwise.
     */
    @JvmStatic
    fun isNetworkEnabled(network: AdNetwork): Boolean {
        logAATKitCall("CMD: isNetworkEnabled($network)")
        return SupportedNetworks.isNetworkEnabled(network)
    }

    /**
     * Get option from AATKit. Options can be obtained from the server or set using the [setOption] method.
     *
     * @param optionName The name of the option to be checked.
     * @return Value of the option or null if it is not set.
     */
    @JvmStatic
    fun getOption(optionName: String): String? {
        logAATKitCall("CMD: getOption($optionName)")
        return AdController.getOption(optionName)
    }

    /**
     * Convenience method for checking if option is enabled in AATKit. Options can be obtained from the server or set using the [setOption] method.
     *
     * @param optionName The name of the option to be checked.
     * @return True if option value is "Yes", false otherwise.
     */
    @JvmStatic
    fun isOptionEnabled(optionName: String): Boolean {
        logAATKitCall("CMD: isOptionEnabled($optionName)")
        return AdController.isOptionEnabled(optionName)
    }

    /**
     * Set option in AATKit. Options can also be obtained from the server.
     *
     * @param optionName  The name of the option to be set.
     * @param optionValue The value of the option to be set.
     */
    @JvmStatic
    fun setOption(optionName: String, optionValue: String) {
        logAATKitCall("CMD: setOption($optionName,$optionName)")
        adController?.setOption(optionName, optionValue) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Checks if consent is of opt-in type (like GDPR) or not
     */
    @JvmStatic
    fun isConsentOptIn(): Boolean {
        logAATKitCall("CMD: isConsentOptIn()")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return false
        }
        return adController.isConsentOptIn()
    }

    @JvmStatic
    fun setRewardedAdSSVInfo(info: RewardedAdSSVInfo) {
        logAATKitCall("CMD: setRewardedAdSSVInfo($info)")
        adController?.setRewardedAdSSVInfo(info) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Sets the targeting information for the application. This information will be used only if no placement-specific targeting is available.
     *
     * @param info Map with targeting information
     * @see .setTargetingInfo
     */
    @JvmStatic
    fun setTargetingInfo(info: Map<String, List<String>>?) {
        logAATKitCall("CMD: setTargetingInfo($info)")
        adController?.setTargetingInfo(info) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Sets the targeting information for the application.
     *
     * @param userTargeting [AATKitUserTargeting]
     * @see .setUserTargeting
     */
    @JvmStatic
    fun setUserTargeting(userTargeting: AATKitUserTargeting) {
        logAATKitCall("CMD: setUserTargeting($userTargeting)")
        adController?.setUserTargeting(userTargeting) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Sets the content targeting url for the application. This information will be used only if no placement-specific targeting is available.
     *
     * @param targetingUrl The targeting url
     * @see .setContentTargetingUrl
     */
    @JvmStatic
    fun setContentTargetingUrl(targetingUrl: String) {
        logAATKitCall("CMD: setContentTargetingUrl($targetingUrl)")
        adController?.setContentTargetingUrl(targetingUrl) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Adds an ad network to the list of ad networks that receive targeting keywords (if any set).
     * If no ad networks are added, any set keywords will be delivered to all ad networks supporting keyword targeting.
     *
     * @param network Chosen ad network.
     */
    @JvmStatic
    fun addAdNetworkForKeywordTargeting(network: AdNetwork) {
        logAATKitCall("CMD: addAdNetworkForKeywordTargeting($network)")
        adController?.addAdNetworkForKeywordTargeting(network) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Removes an ad network from the list of ad networks that receive targeting keywords (if any set).
     * If no ad networks are added to the list, any set keywords will be delivered to all ad networks supporting keyword targeting.
     *
     * @param network Chosen ad network.
     */
    @JvmStatic
    fun removeAdNetworkForKeywordTargeting(network: AdNetwork) {
        logAATKitCall("CMD: removeAdNetworkForKeywordTargeting($network)")
        adController?.removeAdNetworkForKeywordTargeting(network) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Allows to set log level from code.
     *
     * @param logLevel Desired log level, as in [android.util.Log] class.
     */
    @JvmStatic
    fun setLogLevel(logLevel: Int) {
        logAATKitCall("CMD: setLogLevel($logLevel)", true)
        adController?.setLogLevel(logLevel) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Returns the log level used by AATKit.
     * Note that the used log level does not have to come from [setLogLevel], as it can also be influenced by server configs and `adb shell setprop` method.
     */
    @JvmStatic
    fun getLogLevel(): Int {
        logAATKitCall("CMD: getLogLevel()")
        return AdController.getLogLevel()
    }

    /**
     * Allows to set if app should be treated as child-directed.
     *
     * @param isChildDirected True if app is child directed, false otherwise.
     */
    @JvmStatic
    fun setIsChildDirected(isChildDirected: Boolean) {
        logAATKitCall("CMD: setIsChildDirected($isChildDirected)")
        adController?.setChildDirect(isChildDirected) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Allows to mute video ads.
     *
     * @param mute True to enable, false to disable ad muting.
     */
    @JvmStatic
    fun muteVideoAds(mute: Boolean) {
        logAATKitCall("CMD: muteVideoAds($mute)")
        adController?.muteVideoAds(mute) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Used to configure the debug screen presentation
     */
    @JvmStatic
    fun configureDebugScreen(configuration: AATKitDebugScreenConfiguration) {
        logAATKitCall("CMD: configureDebugScreen($configuration)")
        adController?.configureDebugScreen(configuration) ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    /**
     * Used for obtaining debug information (the same that would be presented in dialog after shaking the device if debug screen is enabled)
     *
     * @return AATKit debug info.
     */
    @JvmStatic
    fun getDebugInfoObject(): AATKitDebugInfo? {
        logAATKitCall("CMD: getDebugInfoObject()")
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }
        return adController.getDebugInfo()
    }

    /**
     * Shows AATKit debug screen.
     */
    @JvmStatic
    fun showDebugDialog() {
        logAATKitCall("CMD: showDebugDialog()")
        adController?.showDebugDialog() ?: w(AATKit::class.java, { "AATKit is not initialized" })
    }

    //region non-public methods
    private fun logAATKitCall(call: String, ignoringLogLevel: Boolean = false) {
        if (ServerLogger.shouldLog(ServerLogger.Event.LOGCMD)) {
            log(call)
        }
        if (Logger.isLoggable(Log.INFO) || ignoringLogLevel) {
            val command = call.replaceFirst("^CMD:\\s*".toRegex(), "")
            i(AATKit::class.java, { command }, ignoringLogLevel)
        }
    }

    internal fun createBannerPlacementForCache(
        placementName: String,
        configuration: BannerConfiguration?,
        cache: BannerCacheImplementation,
        infeedPlacementData: InfeedPlacementData
    ): Pair<InfeedBannerPlacementImplementation, Activity>? {
        val adController = adController
        if (adController == null) {
            w(AATKit::class.java, { "AATKit is not initialized" })
            return null
        }

        return adController.createBannerPlacementForCache(
            placementName = placementName,
            configuration = configuration,
            cache = cache,
            infeedPlacementData = infeedPlacementData
        )
    }

    // methods below are intended only for use by the plugins.
    @JvmStatic
    fun destroy() {
        adController?.destroy()
        adController = null
    }

    @JvmStatic
    val isInitialized: Boolean
        get() = adController != null

    //endregion

    /**
     * Defines possible positions for AdChoices icon for Google native ads.
     */
    enum class AdChoicesIconPosition {
        TOP_LEFT, TOP_RIGHT, BOTTOM_LEFT, BOTTOM_RIGHT
    }

    /**
     * Notifies about AATKit events.
     */
    interface Delegate {
        /**
         * Notifies that the AATKit has obtained ad rules.
         *
         * @param fromTheServer Indicates if the rules came from the server. It will return false if currently used rules come from the [AATKitConfiguration.initialRules] or the cached rules are used.
         */
        fun aatkitObtainedAdRules(fromTheServer: Boolean)

        /**
         * Notifies that application's bundle ID was not recognized by the AddApptr server.
         */
        fun aatkitUnknownBundleId()
    }

    /**
     * Notifies about placement reporting events, like counted adspace, request etc.
     */
    interface StatisticsListener {
        /**
         * Notifies that an adspace has been counted.
         */
        fun countedAdSpace(placement: Placement)

        /**
         * Notifies than an mediation cycle has been counted.
         */
        fun countedMediationCycle(placement: Placement)

        /**
         * Notifies that an request has been counted for a given network.
         *
         * @param network Network for which the request has been counted.
         */
        fun countedRequest(placement: Placement, network: AdNetwork)

        /**
         * Notifies that an response has been counted for a given network.
         *
         * @param network Network for which the response has been counted.
         */
        fun countedResponse(placement: Placement, network: AdNetwork)

        /**
         * Notifies that an impression has been counted for a given network.
         *
         * @param network Network for which the impression has been counted.
         */
        fun countedImpression(placement: Placement, network: AdNetwork)

        /**
         * Notifies that an viewable impression has been counted for a given network.
         *
         * @param network Network for which the viewable impression has been counted.
         */
        fun countedVimpression(placement: Placement, network: AdNetwork)

        /**
         * Notifies that an network impression has been counted for a given network.
         *
         * @param network Network for which the network impression has been counted.
         */
        fun countedNimpression(placement: Placement, network: AdNetwork)

        /**
         * Notifies that an direct deal impression has been counted for a given network.
         *
         * @param network Network for which the direct deal impression has been counted.
         */
        fun countedDirectDealImpression(placement: Placement, network: AdNetwork)

        /**
         * Notifies that an click has been counted for a given network.
         *
         * @param network Network for which the click has been counted.
         */
        fun countedClick(placement: Placement, network: AdNetwork)
    }

}

/**
 * Used for passing of detailed GDPR consent. Possible implementations: [ManagedConsent], [SimpleConsent], [VendorConsent].
 */
interface Consent {
    /**
     * Sets to set the networks that will not be requested if they do not have the TCF2 consent. Only works if TCF2 compliant CMP is used.
     *
     * @param stopSet Set of networks not to be requested without TCF2 consent.
     */
    fun setNoConsentNetworkStopSet(stopSet: Set<AdNetwork>?)
}

/**
 * Notifies about display reporting events.
 */
interface AdDisplayListener {
    /**
     * Notifies that ad went fullscreen and that application should pause.
     */
    fun onPauseForAd(placement: Placement)

    /**
     * Notifies that ad came back from fullscreen and that application should resume.
     */
    fun onResumeAfterAd(placement: Placement)

}

/**
 * Notifies about have ad events.
 */
interface HaveAdListener {
    /**
     * Notifies that placement has finished loading an ad successfully.
     */
    fun onHaveAd(placement: Placement)
}

/**
 * Notifies about no ad events.
 */
interface NoAdListener {
    /**
     * Notifies that placement has failed to load an ad.
     */
    fun onNoAd(placement: Placement)
}

/**
 * Notifies about Sticky banner placement events
 */
interface StickyBannerPlacementListener : AdDisplayListener, HaveAdListener, NoAdListener

/**
 * Notifies about MultiSizeBanner placement events
 */
interface MultiSizeBannerPlacementListener : AdDisplayListener, NoAdListener {
    /**
     * Notifies that AATKit has new banner view ready for placement.
     *
     * @param bannerView  Loaded banner view
     */
    fun onHaveAdWithBannerView(placement: Placement, bannerView: BannerPlacementLayout)
}

/**
 * Notifies about AutoLoad banner placement events
 */
interface AutoLoadBannerPlacementListener : AdDisplayListener, HaveAdListener, NoAdListener

/**
 * Notifies about AutoLoad MultiSIze banner placement events
 */
interface AutoLoadMultiSizeBannerPlacementListener : AdDisplayListener, NoAdListener {
    /**
     * Notifies that AATKit has new banner view ready for placement.
     *
     * @param bannerView  Loaded banner view
     */
    fun onHaveAdWithBannerView(placement: Placement, bannerView: BannerPlacementLayout)
}

/**
 * Notifies about InFeed banner placement events
 */
interface InfeedBannerPlacementListener : AdDisplayListener

/**
 * Notifies about Fullscreen placement events
 */
interface FullscreenPlacementListener : AdDisplayListener, HaveAdListener, NoAdListener

/**
 * Notifies about AppOpen placement events
 */
interface AppOpenPlacementListener : FullscreenPlacementListener

/**
 * Notifies about RewardedVideo placement events
 */
interface RewardedVideoPlacementListener : FullscreenPlacementListener {
    /**
     * Notifies that placement has earned incentive.
     *
     * @param aatKitReward Incentive object. Can be null for networks not supporting reward information.
     */
    fun onUserEarnedIncentive(placement: Placement, aatKitReward: AATKitReward?)
}

/**
 * Notifies about Native placement events
 */
interface NativePlacementListener : AdDisplayListener, HaveAdListener, NoAdListener