package com.intentsoftware.addapptr.consent

import android.app.Activity
import android.content.Context
import com.intentsoftware.addapptr.AATKit
import com.intentsoftware.addapptr.AdNetwork
import com.intentsoftware.addapptr.ManagedConsent
import com.intentsoftware.addapptr.NonIABConsent
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.intentsoftware.addapptr.internal.module.Logger
import com.intentsoftware.addapptr.internal.module.Logger.w
import com.sfbx.appconsentv3.AppConsent
import com.sfbx.appconsentv3.ui.AppConsentSDK
import com.sfbx.appconsentv3.ui.listener.OnPresentNoticeListener
import com.sfbx.appconsentv3.ui.model.ACConfiguration
import com.sfbx.appconsentv3.ui.model.ACConsentStatus

/**
 * Wrapper for SFBX CMP
 * @param activity Your Activity instance
 */
class CMPAppConsent(
    activity: Activity,
    private val appKey: String
) : CMPImplementation() {

    private var delegate: CMPDelegate? = null
    private var enabledAdditionalPartners: List<String>? = null

    private var appConsent: AppConsent? = null

    override fun showIfNeeded(activity: Activity) {
        tryToDisplayCmpAndCheckUpdateIfNeeded()
    }

    override fun editConsent(activity: Activity) {
        if (appConsent != null) {
            setOnPresentNoticeListener()
            if (appConsent?.tryToDisplayNotice(force = true) == true) {
                delegate?.didShowCMP()
            }
        } else {
            delegate?.onCMPFailedToShow("Consent form not ready")
        }
    }

    override fun setDelegate(delegate: CMPDelegate) {
        this.delegate = delegate
    }

    override fun reload(activity: Activity) {
    }

    // see https://storage.googleapis.com/tcfac/additional-consent-providers.csv for partners list
    public override fun getConsentForNetwork(network: AdNetwork): NonIABConsent {
        return when (network) {
            AdNetwork.APPLOVIN, AdNetwork.APPLOVINMAX -> readConsentForProviderId(1301)
            AdNetwork.FACEBOOK -> readConsentForProviderId(89)
            AdNetwork.PUBNATIVE -> readConsentForProviderId(2977)
            AdNetwork.IRONSOURCENEW -> readConsentForProviderId(2878)
            AdNetwork.UNITY -> readConsentForProviderId(3234)
            AdNetwork.VUNGLE2 -> readConsentForProviderId(2707)
            AdNetwork.KIDOZ -> readConsentForProviderId(3183)
            else -> {
                //FIXME: missing mappings for Huawei, InMobi
                w(this, { "No mapping for network $network available, treating consent as unknown" })
                NonIABConsent.UNKNOWN
            }
        }
    }

    private fun initCmpModule() {
        if (!AppConsentSDK.isSdkInitialized()) {
            val acConfiguration = ACConfiguration.Builder()
                .setForceApplyGDPR(true)
                .setFullScreenMode(true)
                .setNeedToDisplayValidationButtonsVertically(isNeedToDisplayButtonsAtVertical = true)
                .build()

            AppConsentSDK.initialize(
                appKey = appKey,
                configuration = acConfiguration
            ) { appConsentInitialized ->
                appConsent = appConsentInitialized
                setOnPresentNoticeListener()
                delegate?.CMPNeedsUI()
            }
        } else {
            if (appConsent == null) {
                appConsent = AppConsentSDK.getInstance()
            }
        }
    }

    private fun removeCMPCallback() {
        appConsent?.setOnPresentNoticeListener(null)
    }

    private fun tryToDisplayCMP(): Boolean {
        val shown = appConsent?.tryToDisplayNotice(false) ?: false
        if (shown) {
            delegate?.didShowCMP()
        }
        return shown
    }

    private fun setOnPresentNoticeListener() {
        appConsent?.setOnPresentNoticeListener(object : OnPresentNoticeListener {
            override fun presentConsentError(error: Throwable?) {
                delegate?.onCMPFailedToShow(error?.localizedMessage ?: "Failed to present")
                removeCMPCallback()
            }

            override fun presentConsentGiven() {
                AATKit.adController?.application?.let {
                    updateMorePartnersData(it)
                }
                val state: ManagedConsent.ManagedConsentState = if (appConsent?.userAcceptAll() == true) {
                    ManagedConsent.ManagedConsentState.OBTAINED
                } else {
                    val obtained = appConsent?.getAllConsentables()?.filter { it.status == ACConsentStatus.ALLOWED}
                    if (obtained?.isNotEmpty() == true) {
                        ManagedConsent.ManagedConsentState.CUSTOM
                    } else {
                        ManagedConsent.ManagedConsentState.WITHHELD
                    }
                }

                delegate?.onConsentUpdated(state)
                removeCMPCallback()
            }
        })
    }

    private fun tryToDisplayCmpAndCheckUpdateIfNeeded() {
        if (!tryToDisplayCMP()) {
            checkIfNoticeHasBeenUpdated()
        }
    }

    private fun checkIfNoticeHasBeenUpdated() {
        appConsent?.checkForUpdate({ isNeedToPresentTheCmp: Boolean ->
            if (isNeedToPresentTheCmp) {
                appConsent?.clearConsent()
                tryToDisplayCMP()
            }
        }) { _: Throwable? ->

        }
    }

    private fun updateMorePartnersData(context: Context) {
        var morePartnersConsentString =
            context.getSharedPreferences(
                context.packageName + "_preferences",
                Context.MODE_PRIVATE
            ).getString(
                MORE_PARTNERS_STRING_KEY, null
            )

        enabledAdditionalPartners = null
        if (morePartnersConsentString != null) {
            morePartnersConsentString =
                morePartnersConsentString.replaceFirst(".*~".toRegex(), "") //remove "1~" or similar
            val partnersArray = morePartnersConsentString.split("\\.".toRegex()).toTypedArray()
            enabledAdditionalPartners = listOf(*partnersArray)
        } else {
            Logger.v(this, { "No more partners string found" })
        }
    }

    private fun readConsentForProviderId(id: Int): NonIABConsent {
        return if (enabledAdditionalPartners != null) {
            if (enabledAdditionalPartners!!.contains(id.toString())) {
                NonIABConsent.OBTAINED
            } else {
                NonIABConsent.WITHHELD
            }
        } else {
            Logger.w(this, { "Enabled additional partners list is not available" })
            NonIABConsent.UNKNOWN
        }
    }

    companion object {
        private const val MORE_PARTNERS_STRING_KEY = "IABTCF_AddtlConsent"
    }

    init {
        if (checkRequiredClasses(
                "com.sfbx.appconsentv3.AppConsent",
                "com.sfbx.appconsentv3.ui.AppConsentSDK",
                "com.sfbx.appconsentv3.ui.listener.OnPresentNoticeListener",
                "com.sfbx.appconsentv3.ui.model.ACConfiguration"
            )
        ) {
            updateMorePartnersData(activity)
            initCmpModule()
            onSuccessfulInitialization()
        }
    }
}