package com.intentsoftware.addapptr

import android.app.Application
import com.intentsoftware.addapptr.internal.AATKitAbstractConfiguration
import com.intentsoftware.addapptr.internal.module.Logger.w

/**
 * Used to set up AATKit.
 *
 * @constructor Creates the default Configuration object to be used when initializing AATKit.
 *
 * @param application Reference to [Application].
 */
class AATKitConfiguration(val application: Application) : AATKitAbstractConfiguration() {
    var initialRules: String? = null
        private set

    /**
     * Allows the AATKit to preserve last downloaded ad rules when the application is closed.
     * Such rules will be re-used next time the application is started, before new ones get downloaded.
     * Enabled by default.
     */
    var isShouldCacheRules = true

    /**
     * Sets if networks without TCF2 consent should be skipped (only works if TCF2 compliant CMP is used). False by default.
     */
    var isShouldSkipRules = false
    private var alternativeBundleId: String? = null

    /**
     * If used together with [setAlternativeBundleId] allows to set if the same fake bundle ID should be used in reporting.
     * True by default. If set to false, real bundle ID will be used in reporting even if fake one is used for testing.
     */
    var isShouldReportUsingAlternativeBundleId = true
    private var testModeAccountId = 0

    /**
     * Sets the delegate notifying about AATKit events.
     */
    var delegate: AATKit.Delegate? = null

    /**
     * Sets if the debug screen should be displayed after shaking the device. Enabled by default.
     */
    var isUseDebugShake = true

    /**
     * Sets the platform. Android by default
     */
    var platform = Platform.ANDROID

    /**
     * Sets the optional listener notifying about reporting.
     */
    var reportsDelegate: ReportsDelegate? = null

    /**
     * Object describing advanced ad-network specific configurations.
     */
    var adNetworkOptions: AATKitAdNetworkOptions? = null

    enum class Platform(val platformPrefix: String, val modelPrefix: String) {
        ANDROID("android.", "Android"), HUAWEI("huawei.", "Huawei");

        override fun toString(): String {
            return "Platform{" +
                    "platformPrefix='" + platformPrefix + '\'' +
                    "modelPrefix='" + modelPrefix + '\'' +
                    '}'
        }
    }

    /**
     * Notifies about placement reporting being sent.
     */
    interface ReportsDelegate {
        /**
         * Notifies that the report has been sent.
         * @param report Report data.
         */
        fun onReportsSent(report: String)
    }

    override fun toString(): String {
        return "AATKitConfiguration{" +
                "consentRequired=" + isConsentRequired +
                ", consent=" + consent +
                ", useGeoLocation=" + isUseGeoLocation +
                ", initialRules='" + initialRules + '\'' +
                ", shouldCacheRules=" + isShouldCacheRules +
                ", alternativeBundleId='" + alternativeBundleId + '\'' +
                ", shouldReportUsingAlternativeBundleId=" + isShouldReportUsingAlternativeBundleId +
                ", testModeAccountId=" + testModeAccountId +
                ", delegate=" + delegate +
                ", useDebugShake=" + isUseDebugShake +
                ", application=" + application +
                ", platform=" + platform +
                ", reportsDelegate=" + reportsDelegate +
                ", adNetworkOptions=" + adNetworkOptions +
                "} " + super.toString()
    }

    /**
     * Allows setting of ad rules that will be used before real rules from the server are downloaded.
     *
     * @param initialRules String containing the rules to be used.
     */
    fun setInitialRules(initialRules: String?) {
        this.initialRules = initialRules
    }

    fun getAlternativeBundleId(): String? {
        return alternativeBundleId
    }

    /**
     * Sets the fake bundle ID for testing purposes. It should only be used during development.
     * Cannot be used together with classic test mode ([setTestModeAccountId]).
     *
     * @param alternativeBundleId Bundle ID to be used during testing.
     */
    fun setAlternativeBundleId(alternativeBundleId: String?) {
        if (this.alternativeBundleId != null) {
            w(this, { "Alternative bundle ID was already set! It will be overriden." })
        }
        if (testModeAccountId != 0) {
            w(this, { "Test mode was already enabled! It will be overriden by this bundle ID." })
        }
        this.alternativeBundleId = alternativeBundleId
    }

    fun getTestModeAccountId(): Int {
        return testModeAccountId
    }

    /**
     * Enables AATKit test ads that should be for testing - only during development.
     * Cannot be used with alternative bundle ID ([setAlternativeBundleId]).
     *
     * @param testModeAccountId Test application id obtained from addapptr.com account.
     */
    fun setTestModeAccountId(testModeAccountId: Int) {
        if (alternativeBundleId != null) {
            w(this, { "Alternative bundle ID was already set! The test mode account ID will be ignored." })
        }
        if (this.testModeAccountId != 0) {
            w(this, { "Test mode was already enabled! Old value for test account ID will be overriden." })
        }
        this.testModeAccountId = testModeAccountId
    }
}