package com.intentsoftware.addapptr

import android.app.Activity
import android.content.Context
import com.intentsoftware.addapptr.ManagedConsent.ManagedConsentDelegate
import com.intentsoftware.addapptr.ManagedConsent.ShowIfNeededSetting
import com.intentsoftware.addapptr.consent.CMP
import com.intentsoftware.addapptr.consent.CMPDelegate
import com.intentsoftware.addapptr.internal.CMPImplementation
import com.intentsoftware.addapptr.internal.ConsentHelper
import com.intentsoftware.addapptr.internal.ConsentImplementation
import com.intentsoftware.addapptr.internal.ConsentTypeForReporting
import com.intentsoftware.addapptr.internal.module.Logger.d
import com.intentsoftware.addapptr.internal.module.Logger.e
import com.intentsoftware.addapptr.internal.module.Logger.i
import com.intentsoftware.addapptr.internal.module.Logger.v
import com.intentsoftware.addapptr.internal.module.Logger.w
import com.intentsoftware.addapptr.internal.module.ManagedConsentManager
import kotlin.math.max

/**
 * Will present a dialog for the users to grant or withhold their consent.
 *
 * @param cmp      Instance of [CMP] to be used.
 * @param context  The [Context] of your application.
 * @param delegate The [ManagedConsentDelegate] instance. Must not be null.
 * @param showIfNeededSetting [ShowIfNeededSetting] setting how the [showIfNeeded] method should behave. [ShowIfNeededSetting.SERVER_SIDE_CONTROL] by default.
 */
class ManagedConsent(
    cmp: CMP,
    context: Context,
    delegate: ManagedConsentDelegate,
    showIfNeededSetting: ShowIfNeededSetting
) : Consent, ConsentImplementation(),
    CMPDelegate {
    private val cmp: CMPImplementation?
    private val delegate: ManagedConsentDelegate?
    private val applicationContext: Context?
    private val showIfNeededSetting: ShowIfNeededSetting?

    /**
     * Will present a dialog for the users to grant or withhold their consent.
     *
     * @param cmp      Instance of [CMP] to be used.
     * @param context  The [Context] of your application.
     * @param delegate The [ManagedConsentDelegate] instance. Must not be null.
     */
    constructor(cmp: CMP, context: Context, delegate: ManagedConsentDelegate) : this(
        cmp,
        context,
        delegate,
        ShowIfNeededSetting.SERVER_SIDE_CONTROL
    )

    /**
     * Presents the consent screen ONLY if it is required by the used CMP (i.e. no user consent has been set yet).
     *
     * @param activity The Activity from which the consent screen will be shown.
     */
    fun showIfNeeded(activity: Activity) {
        i(this, { "showIfNeeded called" })

        when (showIfNeededSetting) {
            ShowIfNeededSetting.ALWAYS -> {
                if (cmp != null && cmp.isSuccessfullyInitialized) {
                    cmp.showIfNeeded(activity)
                } else {
                    e(this, { "ManagedConsent was not initialized with a working CMP. Cannot show." })
                }
            }

            ShowIfNeededSetting.NEVER -> {
                w(this, { "ManageConsent was ignore 'showIfNeeded' call, because of your consent configuration" })
            }

            ShowIfNeededSetting.SERVER_SIDE_CONTROL -> {
                if (ManagedConsentManager.serverNeedConsent()) {
                    if (cmp != null && cmp.isSuccessfullyInitialized) {
                        cmp.showIfNeeded(activity)
                    } else {
                        e(this, { "ManagedConsent was not initialized with a working CMP. Cannot show." })
                    }
                } else {
                    w(this, { "ManageConsent was ignore 'showIfNeeded' call, because of your consent configuration" })
                }
            }

            else -> {
                e(this, { "Passed CMP is not an instance of allowed classes. ManagedConsent will not work." })
            }
        }
    }

    /**
     * Presents the consent screen if it is required by the used CMP (i.e. no user consent has been set yet) or the user has rejected the consent.
     *
     * @param daysAgo Minimum number of days that should be passed before re-showing the CMP
     * @param activity The Activity from which the consent screen will be shown.
     */
    fun showIfNeededOrRejected(daysAgo: Int, activity: Activity) {
        i(this, { "showIfNeededOrRejected with days ago: $daysAgo called" })

        val minimumDays = max(daysAgo, 1)
        d(this, { "Minimum days for re-show: $minimumDays" })

        when (ConsentHelper.shouldReshowCMP(daysAgo)) {
            ConsentHelper.ShouldReshowCMPStatus.SHOW_IF_NEEDED -> {
                d(this, {"Will show CMP"})
                showIfNeeded(activity)
            }
            ConsentHelper.ShouldReshowCMPStatus.EDIT_CONSENT -> {
                d(this, { "Will re-show the CMP" })
                editConsent(activity)
            }
            ConsentHelper.ShouldReshowCMPStatus.DO_NOT_SHOW -> {
                v(this, { "Will not show the CMP" })
            }
        }
    }

    /**
     * Presents the consent screen, allowing the user to change consent settings.
     *
     * @param activity The Activity from which the consent screen will be shown.
     */
    fun editConsent(activity: Activity) {
        i(this, { "editConsent called" })
        if (cmp != null && cmp.isSuccessfullyInitialized) {
            cmp.editConsent(activity)
        } else {
            e(this, { "ManagedConsent was not initialized with a working CMP. Cannot edit." })
        }
    }

    /**
     * Tells the CMP to reload. You can call this method for example after receiving [ManagedConsentDelegate.managedConsentCMPFailedToLoad] callback.
     *
     * @param activity The Activity instance.
     */
    fun reload(activity: Activity) {
        i(this, { "reload called" })
        if (cmp != null && cmp.isSuccessfullyInitialized) {
            cmp.reload(activity)
        } else {
            e(this, { "ManagedConsent was not initialized with a working CMP. Cannot reload." })
        }
    }

    override fun onConsentUpdated(state: ManagedConsentState) {
        if (applicationContext != null) {
            readConsentStringFromSharedPreferences(applicationContext)
            ConsentHelper.reconfigureNetworks(applicationContext)
        } else {
            e(this, { "Cannot handle consent update, application context is null" })
        }
        d(
            this,
            { "Calling managed consent delegate method: managedConsentCMPFinished($state)" }
        )
        delegate?.managedConsentCMPFinished(state)
        // Re-download rules after consent update
        AATKit.adController?.configDownloader?.forceReload()
    }

    override fun onCMPFailedToShow(error: String) {
        w(this, { "Failed to show CMP: $error" })
        d(this, { "Calling managed consent delegate method: managedConsentCMPFailedToShow($this, $error)" })
        delegate?.managedConsentCMPFailedToShow(this, error)
    }

    override fun onCMPFailedToLoad(error: String) {
        w(this, { "Failed to load CMP: $error" })
        d(this, { "Calling managed consent delegate method: managedConsentCMPFailedToLoad($this, $error)" })
        delegate?.managedConsentCMPFailedToLoad(this, error)
    }

    override fun CMPNeedsUI() {
        i(this, { "CMP needs UI" })
        d(this, { "Calling managed consent delegate method: managedConsentNeedsUserInterface($this)" })
        delegate?.managedConsentNeedsUserInterface(this)
    }

    override fun didShowCMP() {
        ConsentHelper.consentStringUpdatedDuringSession = true
    }

    override val consentTypeForReporting: ConsentTypeForReporting
        get() = ConsentTypeForReporting.MANAGED_CONSENT

    override val consentType: String
        get() = "ManagedConsent"
    override val usedConsent: String
        get() = "${cmp?.javaClass?.simpleName}"

    override fun toString(): String {
        return "ManagedConsent{" +
                "cmp=" + cmp +
                ", delegate=" + delegate +
                "} " + super.toString()
    }

    /**
     * Notifies about the need to show the consent dialog.
     */
    interface ManagedConsentDelegate {
        /**
         * Notifies that [ManagedConsent] needs to show a consent dialog. After receiving this notification, you should pause your application and call the [showIfNeeded] method.
         *
         * @param managedConsent The [ManagedConsent] instance that requests UI.
         */
        fun managedConsentNeedsUserInterface(managedConsent: ManagedConsent)

        /**
         * Notifies that the used [CMP] has finished updating consent.
         * @param state The [ManagedConsentState] informing about consent given by the user.
         */
        fun managedConsentCMPFinished(state: ManagedConsentState)

        /**
         * Notifies that the used [CMP] failed to load.
         *
         * @param managedConsent @param managedConsent The [ManagedConsent] instance.
         * @param error          The description of what went wrong
         */
        fun managedConsentCMPFailedToLoad(managedConsent: ManagedConsent, error: String?)

        /**
         * Notifies that the used [CMP] failed to show.
         *
         * @param managedConsent @param managedConsent The [ManagedConsent] instance.
         * @param error          The description of what went wrong
         */
        fun managedConsentCMPFailedToShow(managedConsent: ManagedConsent, error: String?)
    }

    /**
     * Possible states of consent given by the user.
     */
    enum class ManagedConsentState {
        /**
         * No information about consent state.
         */
        UNKNOWN,

        /**
         * Consent has been declined by the user.
         */
        WITHHELD,

        /**
         * Partial consent has been granted by the user - at least some purposes and some vendors were given consent.
         */
        CUSTOM,

        /**
         * Full consent has been granted by the user.
         */
        OBTAINED
    }

    enum class ShowIfNeededSetting {
        ALWAYS,
        NEVER,
        SERVER_SIDE_CONTROL
    }

    init {
        if (cmp is CMPImplementation) {
            if (cmp.isSuccessfullyInitialized) {
                this.cmp = cmp
                this.cmp.setDelegate(this)
                this.delegate = delegate
                this.applicationContext = context.applicationContext
                this.showIfNeededSetting = showIfNeededSetting
                if (showIfNeededSetting == ShowIfNeededSetting.SERVER_SIDE_CONTROL) {
                    ManagedConsentManager.setCallbackToConsentNeedsUi {
                        CMPNeedsUI()
                    }
                }
                d(this, { "initialized ManagedConsent: " + toString() })
            } else {
                e(this, { "Passed CMP was not successfully initialized. ManagedConsent will not work." })
                this.cmp = null
                this.delegate = null
                this.applicationContext = null
                this.showIfNeededSetting = null
            }
        } else {
            e(this, { "Passed CMP is not an instance of allowed classes. ManagedConsent will not work." })
            this.cmp = null
            this.delegate = null
            this.applicationContext = null
            this.showIfNeededSetting = null
        }
    }
}