package com.intentsoftware.addapptr

enum class ImpressionPricePrecisionType {
    UNKNOWN,
    ESTIMATED,
    FLOOR_PRICE,
    PUBLISHER_PROVIDED,
    PRECISE
}

/**
 * Object containing impression level information.
 */
@Suppress("unused", "MemberVisibilityCanBePrivate")
class AATKitImpression internal constructor(
    private val bannerSize: String?,
    val adNetwork: AdNetwork,
    val networkKey: String,
    val isDirectDeal: Boolean,
    val mediationType: MediationType,
    val price: Double,
    /**
     * Returns the currency code as string.
     *
     * @return String describing the currency code, or null.
     */
    val currencyCode: String? = when (mediationType) {
        MediationType.MAYO -> "EUR"
        MediationType.AUCTION -> "USD"
        else -> null
    },
    /**
     * Returns the price precision type as [ImpressionPricePrecisionType].
     *
     * @return Impression Price Precision Type.
     */
    val precision: ImpressionPricePrecisionType = when (mediationType) {
        MediationType.MAYO -> ImpressionPricePrecisionType.FLOOR_PRICE
        MediationType.AUCTION -> ImpressionPricePrecisionType.PRECISE
        else -> ImpressionPricePrecisionType.UNKNOWN
    }
) {
    /**
     * Returns the banner size as string (matching the ad key configuration), or null for non-banner ads
     *
     * @return String describing banner size or null for non-banner ads.
     */
    fun getBannerSize(): String? {
        return bannerSize
    }

    /**
     * Returns the network name as string, matching the ad key configuration.
     *
     * @return String describing network name.
     */
    val adNetworkName: String
        get() = adNetwork.name

    /**
     * Returns the mediation type as string.
     *
     * @return String describing the mediation type.
     */
    fun getMediationTypeName(): String {
        return mediationType.name
    }

    override fun toString(): String {
        return "AATKitImpression{" +
                "bannerSize=" + bannerSize +
                ", adNetwork=" + adNetwork +
                ", networkKey='" + networkKey + '\'' +
                ", isDirectDeal=" + isDirectDeal +
                ", mediationType=" + mediationType +
                ", price=" + price +
                ", currencyCode=" + currencyCode +
                ", precision=" + precision +
                '}'
    }
}