package com.intentsoftware.addapptr

import android.graphics.Bitmap
import android.view.View
import com.intentsoftware.addapptr.AATKit.setTargetingInfo

/**
 * Placement used for obtaining banner ads.
 */
@Suppress("unused")
interface StickyBannerPlacement : Placement {
    /**
     * Set the [placement listener][StickyBannerPlacementListener] that will listen to ad loading and display events.
     */
    var listener: StickyBannerPlacementListener?

    /**
     * Sets the [statistics listener][AATKit.StatisticsListener] that will be notified about placement reporting events, like counted adspace, request etc.
     */
    var statisticsListener: AATKit.StatisticsListener?

    /**
     * Sets the [impression listener][ImpressionListener] for the given placement.
     */
    var impressionListener: ImpressionListener?

    /**
     * Setting for Google collapsible banners. Default is null, meaning disabled support for collapsible banners.
     */
    var collapsibleBannerOptions: CollapsibleBannerOptions?

    /**
     * Enables automatic reloading of placement. Autoreloader will use reload time configured on addapptr.com account or fallback to default {@value #BANNER_DEFAULT_RELOAD_INTERVAL_IN_SECONDS} seconds.
     */
    fun startAutoReload()

    /**
     * Enables automatic reloading of placement and sets custom reload time. This reload time will be used instead of time configured on addapptr.com account.
     *
     * @param seconds     Interval between ad reloads. Should be a value from
     * 30
     * up to
     * 600 seconds.
     */
    fun startAutoReload(seconds: Int)

    /**
     * Disables automatic reloading of placement.
     */
    fun stopAutoReload()

    /**
     * Sets reload time that will be used instead of time configured on addapptr.com account.
     *
     * @param seconds     Interval between ad reloads. Should be a value from {@value #BANNER_MIN_RELOAD_INTERVAL_IN_SECONDS} to {@value #BANNER_MAX_RELOAD_INTERVAL_IN_SECONDS} seconds.
     */
    fun setAutoreloadInterval(seconds: Int)

    /**
     * Requests placement reload. Works only if automatic reloading is disabled.
     *
     * @return True if call ends in causing a placement to reload, false otherwise.
     */
    fun reload(): Boolean

    /**
     * Requests placement reload. Works only if automatic reloading is disabled.
     *
     * @param force   True if you want to be able to reload and show new banner before set banner reload interval (default: 30s) between reloads, false otherwise.
     * @return True if call ends in causing a placement to reload, false otherwise.
     */
    fun reload(force: Boolean = false): Boolean

    /**
     * Returns true if there is an ad loaded for given placementId.
     *
     * @return True if there is an ad loaded for given placementId.
     */
    fun hasAd(): Boolean

    /**
     * Returns placement view. Works only for banner placements.
     *
     * @return Placement view or null.
     */
    fun getPlacementView(): View?

    /**
     * Sets placement default image. This image will be shown in placement when no ad is available.
     *
     * @param image       The bitmap to set.
     */
    fun setDefaultImage(image: Bitmap?)

    /**
     * Sets placement default image. This image will be shown in placement when no ad is available.
     *
     * @param resId       The identifier of the resource.
     */
    fun setDefaultImage(resId: Int)

    /**
     * Sets gravity for ads that don't fill entire placement area. Works only for banner placements.
     *
     * @param gravity     The [android.view.Gravity] to set.
     */
    fun setContentGravity(gravity: Int)

    /**
     * Sets the targeting information for the given placement. Information provided for placement overrides targeting information for application set by the [setTargetingInfo].
     */
    var targetingInfo: Map<String, List<String>>?

    /**
     * Sets the content targeting url for the application. This information will be used only if no placement-specific targeting is available.
     *
     * @see .setContentTargetingUrl
     */
    var contentTargetingUrl: String?
}